# -*- coding: utf-8 -*-
from datetime import timedelta, datetime

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.import_marker import AviaImportMarker
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerTransfer, MarkerWriter


class FlyoneMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://reportapi.flyone.eu/api/statistics'

    def __init__(self, login, password, geo_point_cache, logger):
        super(FlyoneMarkerReader, self).__init__(
            statuses_map={
                'CANCELLED': 'cancel',
                'FAILED': 'cancel',
                'PAID': 'paid',
            },
            logger=logger,
            geo_point_cache=geo_point_cache,
            ssl_verify=False,
        )
        self._login = login
        self._password = password

    def parse_report(self, content):
        import ujson

        orders = []
        try:
            response = ujson.loads(content)
            bookings = response['bookings']['booking'] if response.get('bookings') else []
        except ValueError:
            self._logger.exception('Bad Flyone JSON')
            return []

        for booking in bookings:
            try:
                flights = [
                    {
                        'from': flight.get('departure'),
                        'to': flight.get('arrival'),
                        'departure_dt': datetime.strptime(
                            '{} {}'.format(flight.get('departureDate'), flight.get('departureTime')),
                            '%Y-%m-%d %H:%M',
                        ),
                        'arrival_dt': datetime.strptime(
                            '{} {}'.format(flight.get('arrivalDate'), flight.get('arrivalTime')),
                            '%Y-%m-%d %H:%M',
                        ),
                    }
                    for flight in booking['segment']['flight']
                ]
                order = {
                    'order_id': booking.get('pnr'),
                    'created_at': self._convert_dt(booking.get('created_at')),
                    'price': float(booking.get('price')),
                    'currency': booking.get('currency'),
                    'status': self._parse_status(booking.get('state')),
                    'marker': booking.get('marker'),
                    'flights': flights,
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        self._fillin_airports(orders)
        return orders

    @staticmethod
    def _convert_dt(raw_dt):
        return datetime.strptime(raw_dt, '%Y-%m-%d %H:%M:%S') if raw_dt else None

    def _get_request_auth(self):
        return None

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'partner': self._login,
                'password': self._password,
                'date1': report_date.strftime('%Y-%m-%d'),
                'date2': report_date.strftime('%Y-%m-%d'),
            }
        )


class AviaImportFlyoneMarker(AviaImportMarker):
    """ Import marker from Flyone """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('ujson'),
        )

    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Import parameters') as import_block:
            partner_code = sdk2.parameters.String('Partner\'s code', required=True, default='flyone')
            source = sdk2.parameters.String('Source', required=True, default='flyone')
            login = sdk2.parameters.String('Login', required=True, default='yandex')
            secret_with_password = sdk2.parameters.YavSecret(
                'Secret with password',
                default='sec-01fbkb96t9byc2n5zwbvc88jk3',
                required=True,
            )

    def on_execute(self):
        password = self.Parameters.secret_with_password.data()['report-api-password-production']
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=FlyoneMarkerReader(self.Parameters.login, password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
