# -*- coding: utf-8 -*-
from datetime import datetime, timedelta
import string
import StringIO

from sandbox import sdk2
from sandbox.projects.avia.import_marker import AviaImportMarker
from sandbox.projects.avia.lib.marker import BaseMarkerReader, MarkerTransfer, MarkerWriter
from sandbox.sandboxsdk.environments import PipEnvironment


class KiwiMarkerReader(BaseMarkerReader):
    DATA_FILE_TEMPLATE = u'yandexflights_daily_{date}_{date}.xlsx'

    def __init__(self, geo_point_cache, ftp_host, login, sshkey, logger):
        super(KiwiMarkerReader, self).__init__(
            statuses_map={
                'closed': 'paid',
                'confirmed': 'paid',
                'refunded': 'cancel',
            },
            logger=logger,
            geo_point_cache=geo_point_cache,
        )
        self._ftp_host = ftp_host
        self._login = login
        self._sshkey = sshkey

    def import_data(self, date):
        import openpyxl
        import paramiko
        import six
        import sys
        import tempfile
        import traceback

        eur_rate = KiwiMarkerReader._fetch_eur_rate()

        with paramiko.SSHClient() as ssh_client:
            ssh_client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
            try:
                pkey = paramiko.RSAKey.from_private_key(StringIO.StringIO(self._sshkey))
                ssh_client.connect(hostname=self._ftp_host, username=self._login, pkey=pkey)
                with ssh_client.open_sftp() as sftp:
                    # Define the file that you want to download from the remote directory
                    remoteFilePath = self.DATA_FILE_TEMPLATE.format(date=date.strftime('%Y-%m-%d'))

                    # List the resulting dictionary keys to output data and the functions to get that data
                    cell_processors = {
                        u'bid': ('order_id', lambda value: str(value)),
                        u'booking_timestamp': ('created_at', KiwiMarkerReader._parse_date),
                        u'price': ('_price_eur', KiwiMarkerReader._to_float),
                        u'external_id': ('marker', KiwiMarkerReader._as_is),
                        u'trip_type': ('trip_type', KiwiMarkerReader._trip_type),
                        u'status': ('status', self._parse_status),
                        u'o_d': ('_o_d', KiwiMarkerReader._as_is),
                    }

                    headers = []
                    with tempfile.NamedTemporaryFile(suffix=".xlsx") as temp:
                        sftp.get(remoteFilePath, temp.name)
                        wb = openpyxl.load_workbook(filename=temp.name)

                        sheet = wb[wb.sheetnames[0]]
                        for index, row in enumerate(sheet.rows):
                            # Skip empty rows (even if as a safety precaution)
                            if all(cell.value is None for cell in row):
                                continue

                            if not bool(headers):
                                headers = [string.strip(cell.value) if cell.value is not None else None
                                           for cell in row]
                                continue

                            order = {}
                            for col_index, cell in enumerate(row):
                                if cell.value is not None:
                                    processor = cell_processors.get(headers[col_index])
                                    if processor:
                                        result = processor[1](cell.value)
                                        if result is not None:
                                            order[processor[0]] = result

                            if order.get('created_at') is None or order.get('marker') is None:
                                continue

                            route = order.get('_o_d')
                            if route is not None and '|' in route:
                                parts = route.split('|', 2)
                                order['airport_from'] = parts[0]
                                order['airport_to'] = parts[1]

                            price_eur = order.get('_price_eur')
                            if price_eur > 0 and eur_rate > 0:
                                order['price'] = KiwiMarkerReader._round(price_eur * eur_rate)

                            yield order
            except IOError:
                self._logger.warn('No such file: %s', remoteFilePath)
            except Exception:
                self._logger.exception(traceback.format_exc())
                exc_info = sys.exc_info()
                six.reraise(ImportError, exc_info[1], exc_info[2])

    def parse_report(self, content):
        return content

    @staticmethod
    def _round(value):
        return int((value * 100) + 0.5) / 100.0

    @staticmethod
    def _as_is(value):
        return value

    @staticmethod
    def _parse_date(value):
        # sample input: 2019-07-16 15:42:26.811175+00
        if not value:
            return None

        if '+' in value:
            value = value.split('+')[0]

        return datetime.strptime(value, '%Y-%m-%d %H:%M:%S.%f')

    @staticmethod
    def _trip_type(value):
        return {
            'oneway': u'oneway',
            'roundtrip': u'roundtrip',
        }.get(value)

    @staticmethod
    def _pop_float(dict, key):
        return KiwiMarkerReader._to_float(dict.pop(key, 0.))

    @staticmethod
    def _to_float(value):
        try:
            return float(value)
        except:
            return 0.

    @staticmethod
    def _fetch_eur_rate():
        # fetch EUR code, the its rate
        import requests
        cur_ref_url = 'http://backend.production.avia.yandex.net/rest/currencies/ru/ru'
        cur_ref_content = requests.get(url=cur_ref_url)
        cur_ref_body = cur_ref_content.json()
        cur_ref = cur_ref_body.get(u'data')

        if cur_ref is None:
            raise ValueError('Unable to fetch the currencies list')

        cur_eur = KiwiMarkerReader._find_item(cur_ref, u'code', u'EUR')
        if cur_eur is None:
            raise ValueError('Unable to find out the ID of the EUR currency')

        cur_eur_id = cur_eur.get(u'id')
        if cur_eur_id is None:
            raise ValueError('EUR currency has no id {eur}'.format(eur=cur_eur))

        # fetch EUR rate
        cur_rate_url = 'http://backend.production.avia.yandex.net/rest/currencies/rates/ru/ru'
        cur_rate_content = requests.get(url=cur_rate_url)
        cur_rate_body = cur_rate_content.json()
        cur_rates_data = cur_rate_body.get(u'data')

        if cur_rates_data is None:
            raise ValueError('Unable to fetch the currency rates data')

        cur_rates = cur_rates_data.get('rates')
        if cur_rates is None:
            raise ValueError('Unable to fetch the currency rates list')

        eur_rate_item = KiwiMarkerReader._find_item(cur_rates, u'currency_id', cur_eur_id)
        if eur_rate_item is None:
            raise ValueError('Unable to find out the EUR currency data')

        eur_rate_obj = eur_rate_item.get('rate')
        if eur_rate_obj is None:
            raise ValueError('Unable to find out the current EUR rate')

        return float(eur_rate_obj)

    @staticmethod
    def _find_item(items, subkey, search_for):
        for item in items:
            if isinstance(item, dict) and item.get(subkey) == search_for:
                return item
        return None


class AviaImportKiwiMarker(AviaImportMarker):
    """ Import marker from Kiwi """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('openpyxl'),
            PipEnvironment('requests'),
        )

    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Import parameters') as import_block:
            partner_code = sdk2.parameters.String('Partner\'s code', required=True, default='kiwi')
            source = sdk2.parameters.String('Source', required=True, default='kiwi')
            ftp_host = sdk2.parameters.String('FTP host', required=True, default='partner-reports.kiwi.com')
            login = sdk2.parameters.String('Login', required=True)

    def on_execute(self):
        login = self.Parameters.login
        sshkey = sdk2.Vault.data(self.Parameters.vaults_owner, 'AVIA_KIWI_SSH_KEY')
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=KiwiMarkerReader(
                geo_point_cache=self.geo_point_cache,
                logger=self._logger,
                ftp_host=self.Parameters.ftp_host,
                login=login,
                sshkey=sshkey,
            ),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
