# -*- coding: utf-8 -*-
import json
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.avia.lib.marker import BatchMarkerReader, MarkerWriter, MarkerBatchTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class NabortuMarkerReader(BatchMarkerReader):
    REQUEST_URL = 'https://media.zelenski.ru/partners/yandex/yandex_report.xml'

    def __init__(self, login, password, geo_point_cache, logger):
        super(NabortuMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'paid': 'paid',
                'cancel': 'cancel',
                'booking': 'booking',
            },
            geo_point_cache=geo_point_cache,
        )
        self._login = login
        self._password = password

    def _get_request_data(self):
        return (
            self.REQUEST_URL,
            {},
        )

    def _get_request_auth(self):
        return self._login, self._password

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring

        orders = []
        try:
            tree = fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad Nabortu XML')
            return []

        for order_tree in tree.xpath('//orders/order'):
            try:
                order_dict = {row.tag: row.text for row in order_tree}
                order = {
                    'marker': order_dict.get('marker'),
                    'status': self._parse_status(order_dict.get('status')),
                    'order_id': order_dict.get('orderid'),
                    'price': order_dict.get('price'),
                    'currency': order_dict.get('currency'),
                    'created_at': self._convert_dt(order_dict.get('created_at').split('.')[0]),
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        return orders

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%dT%H:%M:%S')


class AviaImportNabortuMarker(AviaImportMarker):
    """ Import marker from Nabortu """
    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Nabortu') as nabortu_group:
            credentials = sdk2.parameters.YavSecret('Credentials', default='sec-01ecskr53e0cvgxt0gtynjqcnk', required=True)
            login_key = sdk2.parameters.String('Key in vault for login', default='report-api-login', required=True)
            password_key = sdk2.parameters.String('Key in vault for password', default='report-api-password', required=True)

    def on_execute(self):
        credentials_data = self.Parameters.credentials.data()
        login = credentials_data[self.Parameters.login_key]
        password = credentials_data[self.Parameters.password_key]
        reader = NabortuMarkerReader(login, password, self.geo_point_cache, self._logger)
        marker_transfer = MarkerBatchTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=reader,
            logger=self._logger,
        )

        marker_transfer.transfer()
