# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class NemoMarkerReader(MarkerHTTPReader):
    def __init__(self, login, password, request_url, logger, geo_point_cache):
        """
        :param str login:
        :param str password:
        :param str request_url:
        :param loggin.Logger logger: логгер для проброса сообщений и ошибок
        """
        super(NemoMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'PROCESSING': 'booking',
                'PAID': 'paid',
                'CANCELLED': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._login = login
        self._password = password
        self.REQUEST_URL = request_url

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'go': 'settings/download_metasearch_stat',
                'partner': self._login,
                'date1': report_date.strftime('%Y-%m-%d'),
                'date2': report_date.strftime('%Y-%m-%d'),
                'password': self._password,
            }
        )

    def _get_request_auth(self):
        return None

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        orders = []
        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad NEMO XML. response start with "%s"', content[:40])
            raise

        for order_tree in tree.xpath('//bookings/booking'):
            try:
                order_dict = {row.tag: row.text for row in order_tree}
                flight_dicts = [
                    {
                        'from': flight['departure'],
                        'to': flight['arrival'],
                        'departure_dt': self._convert_flight_dt(
                            flight['departureDate'] + ' ' + flight['departureTime']
                        ),
                        'arrival_dt': self._convert_flight_dt(
                            flight['arrivalDate'] + ' ' + flight['arrivalTime']
                        ),
                    }
                    for flight in
                    [
                        {
                            param.tag: param.text for param in flight_tree
                        }
                        for flight_tree in order_tree.xpath('segment/flight')
                    ]
                ]
                order = {
                    'order_id': order_dict.get('id'),
                    'created_at': self._convert_dt(order_dict.get('created_at')),
                    'price': order_dict.get('price'),
                    'currency': order_dict.get('currency'),
                    'status': self._parse_status(order_dict.get('state')),
                    'marker': order_dict.get('marker'),
                    'flights': flight_dicts,
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        self._fillin_airports(orders)
        return orders

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M:%S')

    @staticmethod
    def _convert_flight_dt(dt_str):
        possible_formats = [
            '%Y-%m-%d %H:%M:%S.%f',
            '%Y-%m-%d %H:%M:%S',
            '%Y-%m-%d %H:%M.%f',
            '%Y-%m-%d %H:%M',
        ]
        errors = []
        for layout in possible_formats:
            try:
                return datetime.strptime(dt_str, layout)
            except ValueError as e:
                errors.append(e)
                continue
        raise ValueError('Cannot parse time with any of layouts: {}'.format(errors))


class AviaImportNemoMarker(AviaImportMarker):
    """ Import marker from nemo-partners """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    class Parameters(AviaImportMarker.Parameters):
        login = sdk2.parameters.String('Partner login', required=True)
        request_url = sdk2.parameters.String('Partner request url', required=True)

    def on_execute(self):
        password = self._get_partner_secret('password')

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=NemoMarkerReader(
                str(self.Parameters.login),
                password,
                str(self.Parameters.request_url),
                self._logger,
                self.geo_point_cache
            ),
            logger=self._logger,
        )

        self._logger.info(
            '%s. Transferring date range: %s - %s',
            self.Parameters.partner_code,
            self._left_date,
            self._right_date
        )
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
