# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class OneTwoTripMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://staging--bo.twiket.com/dashboard/getReportByAuth'
    LOGIN = "stats@yandex"

    def __init__(self, password, geo_point_cache, logger):
        super(OneTwoTripMarkerReader, self).__init__(
            logger=logger,
            statuses_map={'Issued': 'paid', 'Void': 'cancel'},
            geo_point_cache=geo_point_cache,
            timeout=(20, 120),
        )
        self._password = password

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'from': report_date.strftime('%Y%m%d'),
                'to': report_date.strftime('%Y%m%d'),
            },
        )

    def _get_request_auth(self):
        import requests.auth

        return requests.auth.HTTPBasicAuth(self.LOGIN, self._password)

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        orders = []
        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad onetwotrip XML')
            return []

        for order_tree in tree.xpath('/Report/Repor'):
            try:
                order = {
                    'order_id': order_tree.attrib['orderid'],
                    'created_at': self._convert_dt(order_tree.attrib['created_at']),
                    'price': order_tree.attrib['price'],
                    'status': self._parse_status(order_tree.attrib['status']),
                    'marker': order_tree.attrib['marker'],
                    'currency': order_tree.attrib['currency'],
                    'airport_from': order_tree.attrib['from'],
                    'airport_to': order_tree.attrib['to'],
                    'trip_type': self._parse_trip_type(order_tree.attrib['type']),
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        return orders

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%d.%m.%Y')

    def _parse_trip_type(self, trip_type):
        trip_type_map = {'OW': u'oneway', 'RT': u'roundtrip', 'OJ': u'openjaw', '': None}
        if trip_type not in trip_type_map:
            self._logger.warning('Unknown trip_type %s', trip_type)
            return
        return trip_type_map[trip_type]


class AviaImportOneTwoTripMarker(AviaImportMarker):
    """Import marker from onetwotrip"""

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (PipEnvironment('lxml'),)

    def on_execute(self):
        password = self._get_partner_secret('password')
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=OneTwoTripMarkerReader(password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
