# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class OzonMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://www.ozon.travel/my/partners/period_xml.html'
    LOGIN = "avia-info@yandex-team.ru"

    def __init__(self, password, geo_point_cache, logger):
        """
        :type password: str
        """
        super(OzonMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'InProgress': 'booking',
                'Confirmed': 'paid',
                'Cancelled': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._password = password

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'DateStart': report_date.strftime('%Y-%m-%d'),
                'Date': report_date.strftime('%Y-%m-%d'),
            }
        )

    def _get_request_auth(self):
        import requests.auth
        return requests.auth.HTTPBasicAuth(self.LOGIN, self._password)

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        orders = []
        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad OZON XML')
            return []

        for order_tree in tree.xpath('//Report/Orders/Order'):
            try:
                if order_tree.get('OrderType') != 'AIR':
                    continue
                order_dict = {row.tag: row for row in order_tree}
                flight_dicts = [
                    {
                        'from': flight.get('From'),
                        'to': flight.get('To'),
                        'departure_dt': self._convert_dt(flight.get('Date')),
                        'arrival_dt': self._convert_dt(flight.get('Date')),
                    }
                    for flight in order_dict['FlightRoute']
                ]
                order = {
                    'order_id': order_tree.get('Number'),
                    'created_at': self._convert_dt(order_tree.get('CreateDate')),
                    'confirm_date': self._convert_dt(order_tree.get('ConfirmDate')),
                    'price': order_tree.get('Sum'),
                    'status': self._parse_status(order_tree.get('Status')),
                    'marker': order_tree.get('PartnerRef'),
                    'flights': flight_dicts,
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        self._fillin_airports(orders)
        return orders

    def _convert_dt(self, dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%dT%H:%M:%S')


class AviaImportOzonMarker(AviaImportMarker):
    """ Import marker from Ozon """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    def on_execute(self):
        password = self._get_partner_secret('password')
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=OzonMarkerReader(password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
