# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.projects.avia.import_marker import AviaImportMarker
from sandbox.projects.avia.lib.marker import BaseMarkerReader, MarkerTransfer, MarkerWriter
from sandbox.sandboxsdk.environments import PipEnvironment


class PobedaMarkerReader(BaseMarkerReader):
    DATA_FILE_TEMPLATE = 'POBEDA-YANDEX-{date}.xlsx'

    def __init__(self, geo_point_cache, ftp_host, login, password, logger):
        super(PobedaMarkerReader, self).__init__(
            statuses_map={},
            logger=logger,
            geo_point_cache=geo_point_cache,
        )
        self._ftp_host = ftp_host
        self._login = login
        self._password = password

    def import_data(self, date):
        import openpyxl
        import paramiko
        import six
        import string
        import sys
        import tempfile
        import traceback

        with paramiko.SSHClient() as ssh_client:
            ssh_client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
            try:
                ssh_client.connect(
                    hostname=self._ftp_host,
                    username=self._login,
                    password=self._password,
                    look_for_keys=False,
                    allow_agent=False,
                )

                with ssh_client.open_sftp() as sftp:
                    # Define the file that you want to download from the remote directory
                    remoteFilePath = self.DATA_FILE_TEMPLATE.format(date=date.strftime('%Y-%m-%d'))

                    # List the resulting dictionary keys to output data and the functions to get that data
                    cell_processors = {
                        u'Время создания': ('created_at', PobedaMarkerReader._parse_date),
                        u'PNR': ('order_id', PobedaMarkerReader._as_is),
                        u'Статус брони': ('status', lambda value: 'paid' if not value else 'cancel'),
                        u'Маршрут': ('trip_type', PobedaMarkerReader._trip_type),
                        u'marker': ('marker', PobedaMarkerReader._as_is),
                        u'Тариф': ('_tariff', PobedaMarkerReader._to_float),
                        u'Возврат тарифа': ('_tariff_refund', PobedaMarkerReader._to_float),
                        u'Услуги': ('_services', PobedaMarkerReader._to_float),
                        u'Возврат услуг': ('_services_refund', PobedaMarkerReader._to_float),
                        u'Комиссия,руб': ('_commission', PobedaMarkerReader._to_float),
                    }

                    headers = []
                    with tempfile.NamedTemporaryFile(suffix=".xlsx") as temp:
                        sftp.get(remoteFilePath, temp.name)
                        wb = openpyxl.load_workbook(filename=temp.name)

                        for sheet_name in wb.get_sheet_names():
                            sheet = wb[sheet_name]
                            for index, row in enumerate(sheet.rows):
                                # Headers are in the second ro, so skip the first one
                                if all(cell.value is None for cell in row):
                                    continue

                                if not bool(headers):
                                    headers = [string.strip(cell.value) if cell.value is not None else None
                                               for cell in row]
                                    continue

                                order = {}
                                for col_index, cell in enumerate(row):
                                    if cell.value is not None:
                                        processor = cell_processors.get(headers[col_index])
                                        if processor:
                                            result = processor[1](cell.value)
                                            if result is not None:
                                                order[processor[0]] = result

                                if order.get('created_at') is None or order.get('marker') is None:
                                    continue

                                price = max(
                                    0.,
                                    PobedaMarkerReader._net_value(order, '_tariff', '_tariff_refund') +
                                    PobedaMarkerReader._net_value(order, '_services', '_services_refund')
                                )
                                order['price'] = PobedaMarkerReader._round(price)
                                if price == 0:
                                    order['status'] = 'cancel'

                                yield order
            except IOError:
                self._logger.warn('No such file: %s', remoteFilePath)
            except Exception:
                self._logger.exception(traceback.format_exc())
                exc_info = sys.exc_info()
                six.reraise(ImportError, exc_info[1], exc_info[2])

    def parse_report(self, content):
        return content

    @staticmethod
    def _round(value):
        return int((value * 100) + 0.5) / 100.0

    @staticmethod
    def _as_is(value):
        return value

    @staticmethod
    def _parse_date(value):
        return datetime.strptime(value, '%d.%m.%Y %H:%M') if value else None

    @staticmethod
    def _trip_type(value):
        return {
            'OW': u'oneway',
            'RT': u'roundtrip',
        }.get(value)

    @staticmethod
    def _net_value(dict, positive_key, negative_key):
        return PobedaMarkerReader._pop_float(dict, positive_key) - PobedaMarkerReader._pop_float(dict, negative_key)

    @staticmethod
    def _pop_float(dict, key):
        return PobedaMarkerReader._to_float(dict.pop(key, 0.))

    @staticmethod
    def _to_float(value):
        try:
            return float(value)
        except:
            return 0.


class AviaImportPobedaMarker(AviaImportMarker):
    """ Import marker from Pobeda """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('openpyxl'),
        )

    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Import parameters') as import_block:
            partner_code = sdk2.parameters.String('Partner\'s code', required=True, default='pobeda')
            source = sdk2.parameters.String('Source', required=True, default='pobeda')
            ftp_host = sdk2.parameters.String('FTP host', required=True, default='ftp.pobeda.aero')
            login = sdk2.parameters.String('Login', required=True)
            password = sdk2.parameters.String('Password', required=False)

    def on_execute(self):
        login = self.Parameters.login if self.Parameters.login \
            else sdk2.Vault.data(self.Parameters.vaults_owner, 'AVIA_POBEDA_LOGIN')
        password = self.Parameters.password if self.Parameters.password \
            else sdk2.Vault.data(self.Parameters.vaults_owner, 'AVIA_POBEDA_PASSWORD')
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=PobedaMarkerReader(
                geo_point_cache=self.geo_point_cache,
                logger=self._logger,
                ftp_host=self.Parameters.ftp_host,
                login=login,
                password=password,
            ),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
