# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class SSevenMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://msestat.api.s7.ru/api/statistics'
    LOGIN = "Yandex"

    def __init__(self, password, geo_point_cache, logger):
        """
        :type password: str
        """
        super(SSevenMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'BOOKED': 'booking',
                'COMPLETED': 'paid',
                'HISTORIC': 'paid',
                'CANCELLED': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._password = password

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'date1': report_date.strftime('%Y-%m-%d'),
                'date2': report_date.strftime('%Y-%m-%d'),
            }
        )

    def _get_request_auth(self):
        import requests.auth
        return requests.auth.HTTPBasicAuth(self.LOGIN, self._password)

    def parse_report(self, content):
        import json

        result_orders = []
        try:
            response = json.loads(content)
        except Exception:
            self._logger.exception('Bad s_seven json')
            return []

        for order in response['orders']:
            try:
                flights = []
                for air in order['airs']:
                    for route in air['routes']:
                        flights.append({
                            'from': route['origin'],
                            'to': route['destination'],
                            'arrival_dt': self._convert_dt(route['arrivalDate']),
                            'departure_dt': self._convert_dt(route['departureDate']),
                        })
                price = 0
                currency = None
                for payment in order['payments']:
                    total = payment.get('pricing', {}).get('total')
                    if not total:
                        continue
                    fees = ['salePrice', 'serviceFee']
                    for fee in fees:
                        if not total[fee]['amount']:
                            continue
                        if not total[fee]['currency']:
                            raise ValueError('Empty currency for payment with non-zero amount: %s', payment)
                        if not currency:
                            currency = total[fee]['currency']
                        if total[fee]['currency'] != currency:
                            raise ValueError('Two different currencies are used in same order: %s, %s',
                                             currency, total[fee]['currency'])
                        price += total[fee]['amount']

                order = {
                    'order_id': order['number'],
                    'status': self._parse_status(order['status']),
                    'created_at': self._parse_created_at(order['created']),
                    'price': price,
                    'currency': currency,
                    'marker': order.get('partner_marker'),
                    'flights': flights,
                }
                result_orders.append(order)
            except Exception:
                self._logger.exception('Parse error. order: %s', order)
                raise
        self._fillin_airports(result_orders)
        return result_orders

    @staticmethod
    def _parse_created_at(dt_str):
        for dt_format in ('%Y-%m-%dT%H:%M:%S.%f', '%Y-%m-%dT%H:%M:%S.%fZ', '%Y-%m-%dT%H:%M:%SZ', '%Y-%m-%dT%H:%M:%S'):
            try:
                return datetime.strptime(dt_str, dt_format)
            except ValueError:
                continue
        raise ValueError('Unsupported created_at datetime format: {}'.format(dt_str))

    def _convert_dt(self, dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%dT%H:%M:%S')


class AviaImportSSevenMarker(AviaImportMarker):
    """ Import marker from SSeven """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default

    def on_execute(self):
        password = self._get_partner_secret('password')
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=SSevenMarkerReader(password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
