#! -*- coding: utf-8 -*-
import datetime
import hashlib
import traceback
from datetime import timedelta

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerWriter
from sandbox.projects.avia.import_marker import AviaImportMarker


class AviaImportSindbadMarker(AviaImportMarker):
    """ Import marker from Sindbad """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
            PipEnvironment('pytz'),
        )

    LOGIN = 'yandex'
    URL = "http://xml.sindbad.ru/api/bookList?year={}&month={}&day={}"

    def on_execute(self):
        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)

        self._import_sindbad(self._left_date, self._right_date)

        self._logger.info('Done')

    def _import_sindbad(self, left_date, right_date):
        report_date = left_date
        while report_date <= right_date:
            try:
                self._logger.info('Process: %s', report_date.strftime('%Y-%m-%d'))
                marker_writer = MarkerWriter(
                    self.Parameters.source,
                    self._logger,
                    self.Parameters.yt_partner_booking_root,
                    self._yt
                )
                marker_writer.add_rows(self.parse_report_xml(
                    self.load_raw_data(report_date).encode('utf-8'),
                    self._partner
                ))
                marker_writer.write_to_yt(report_date)
            except:
                self._logger.exception(traceback.format_exc())
            report_date += timedelta(days=1)

    @staticmethod
    def get_utc_now():
        import pytz
        return pytz.UTC.localize(datetime.datetime.utcnow())

    @staticmethod
    def get_url_for_date(date):
        return AviaImportSindbadMarker.URL.format(date.strftime('%Y'), date.strftime('%m'), date.strftime('%d'))

    def get_password(self):
        key = self._get_partner_secret('password')
        m = hashlib.md5()
        m.update(''.join([
            'sindbad', self.get_utc_now().strftime('%Y-%m-%d'), key
        ]))

        return m.hexdigest()

    def load_raw_data(self, date):
        import requests.auth

        r = requests.get(
            self.get_url_for_date(date),
            auth=requests.auth.HTTPBasicAuth(self.LOGIN, self.get_password())
        )
        return r.text

    @staticmethod
    def convert_dt(dt_str):
        return datetime.datetime.strptime(dt_str, "%Y-%m-%d %H:%M")

    @staticmethod
    def get_price(price_element):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        attribs = price_element.attrib
        if len(attribs) != 1:
            raise Exception('Multiple prices found: {!r}'.format(etree.tostring(price_element)))
        [(currency, price)] = attribs.items()
        return currency, price

    @staticmethod
    def parse_status(status):
        parsed = {
            'payment_expected': 'booking',
            'processing': 'booking',
            'done': 'paid',
            'paid': 'paid',
            'cancelled': 'cancel',
        }.get(status)
        if status and not parsed:
            raise Exception('Unknown status: {}'.format(status))
        return parsed

    @staticmethod
    def get_segments(flight_element):
        return [seg for seg in flight_element.getchildren() if seg.get('dir') == 'OUT']

    @staticmethod
    def get_airports(segments):
        """
        Получаем из неупорядоченного списка сегментов пункт начала маршрута и пункт конца.
        Проверяем, что это один цельный маршрут.
        """
        segment_dict = {seg.get('orig'): seg.get('dest') for seg in segments}
        if len(segment_dict) != len(segments):
            raise Exception('Repeated departure point')
        if len(set(segment_dict.values())) != len(segments):
            raise Exception('Repeated arrival point')

        if not segment_dict:
            raise Exception('No segments found with direction "OUT"')

        destinations = set(segment_dict.values())
        origins = set(segment_dict.keys())
        try:
            [orig] = origins - destinations
            [dest] = destinations - origins
        except Exception:
            raise Exception("Flights don't form a path")

        node = orig
        for _ in range(len(segment_dict)):
            node = segment_dict.get(node)
        if node != dest:
            raise Exception('Some flights form an independent cycle')

        return orig, dest

    def parse_report_xml(self, xml, partner):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        rows = []
        try:
            tree = safe_fromstring(xml)
        except etree.XMLSyntaxError:
            self._logger.exception("Bad SINDBAD XML")
            return []

        for order in tree.xpath('//result/reserve'):
            try:
                order_dict = {row.tag: row for row in order.getchildren()}
                currency, price = self.get_price(order_dict['price'])
                airport_from, airport_to = self.get_airports(self.get_segments(order_dict['flight']))
                rows.append(MarkerWriter.Row(
                    created_at=self.convert_dt(order.get('created')),
                    marker=order.get('token'),
                    status=self.parse_status(order.get('status')),
                    partner=partner,
                    partner_id=partner['partner_id'],
                    billing_order_id=partner['billing_order_id'],
                    order_id=order.get('pnr'),
                    price=price,
                    currency=currency,
                    airport_from=airport_from,
                    airport_to=airport_to,
                ))
            except Exception:
                self._logger.exception('Error while parsing {!r}'.format(etree.tostring(order)))
        return rows
