# -*- coding: utf-8 -*-
import hashlib
from collections import OrderedDict
from datetime import datetime, timedelta

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class SmartaviaMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://bo-prod-api.flyways.ru/api/meta/report/5n'

    def __init__(self, password, geo_point_cache, logger):
        """
        :param str password: пароль для доступа
        """
        super(SmartaviaMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'PROCESSING': 'booking',
                'PAID': 'paid',
                'CANCELED': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._password = password

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'date1': report_date.strftime('%Y-%m-%d'),
                'date2': report_date.strftime('%Y-%m-%d'),
                'password': self._password,
            }
        )

    def _get_request_auth(self):
        return None

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        orders = []
        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Incorrect Smartavia XML')
            return []

        for booking in tree.xpath('//bookings/booking'):
            try:
                booking_dict = self._to_dict(booking)

                airport_from = ''
                airport_to = ''
                segments = list(booking.xpath('segment'))
                if segments:
                    flights_from = list(segments[0].xpath('flight'))
                    if flights_from:
                        airport_from = self._to_dict(flights_from[0]).get('departure')

                    if len(segments) > 1:
                        flights_to = list(segments[-1].xpath('flight'))
                        if flights_to:
                            airport_to = self._to_dict(flights_to[0]).get('departure')
                    else:
                        airport_to = self._to_dict(flights_from[-1]).get('arrival')

                if not airport_from:
                    self._logger.exception('Unable to parse airport_from')

                if not airport_to:
                    self._logger.exception('Unable to parse airport_to')

                order = {
                    'order_id': booking_dict.get('id'),
                    'created_at': self._convert_dt(booking_dict.get('created_at')),
                    'price': booking_dict.get('price'),
                    'status': self._parse_status(booking_dict.get('state')),
                    'marker': booking_dict.get('marker'),
                    'airport_from': airport_from,
                    'airport_to': airport_to,
                    'currency': booking_dict.get('currency'),
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        return orders

    @staticmethod
    def _to_dict(node):
        return {row.tag: row.text for row in node}

    @staticmethod
    def _convert_dt(dt_str):
        if len(dt_str) > 10:
            dt_str = dt_str[:10]
        return datetime.strptime(dt_str, '%Y-%m-%d')


class AviaImportSmartaviaMarker(AviaImportMarker):
    """ Import marker from Smartavia """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    def on_execute(self):
        password = self._get_partner_secret('password')

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=SmartaviaMarkerReader(password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
