# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class SuperkassaMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://superkassa.net/api/partner/statistics.php'
    LOGIN = 'yandex'

    def __init__(self, password, authhash, geo_point_cache, logger):
        """
        :type password: str
        """
        super(SuperkassaMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'PROCESSING': 'booking',
                'PAID': 'paid',
                'CANCELLED': 'cancel',
                'REFUND': 'cancel',
            },
            geo_point_cache=geo_point_cache,
            timeout=(60.0, 60.0),
        )
        self._password = password
        self._authhash = authhash

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'date1': report_date.strftime('%Y-%m-%d'),
                'date2': report_date.strftime('%Y-%m-%d'),
                'hash': self._authhash,
            }
        )

    def _get_request_auth(self):
        import requests.auth
        return requests.auth.HTTPBasicAuth(self.LOGIN, self._password)

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        orders = []
        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad Superkassa XML')
            return []

        for order_tree in tree.xpath('//booking'):
            try:
                order_dict = {row.tag: row.text for row in order_tree}
                if order_dict.get('state') == 'REFUND':
                    continue
                flight_dicts = [
                    {
                        'from': flight['departure'],
                        'to': flight['arrival'],
                        'departure_dt': self._convert_flight_dt(
                            flight['departureDate'] + ' ' + flight['departureTime']
                        ),
                        'arrival_dt': self._convert_flight_dt(
                            flight['arrivalDate'] + ' ' + flight['arrivalTime']
                        ),
                    }
                    for flight in
                    [
                        {
                            param.tag: param.text for param in flight_tree
                        }
                        for flight_tree in order_tree.xpath('segment/flight')
                    ]
                ]
                order = {
                    'order_id': order_dict.get('id'),
                    'created_at': self._convert_dt(order_dict.get('created_at')),
                    'price': order_dict.get('price'),
                    'currency': order_dict.get('currency'),
                    'status': self._parse_status(order_dict.get('state')),
                    'marker': order_dict.get('marker'),
                    'flights': flight_dicts,
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        self._fillin_airports(orders)
        return orders

    def _convert_dt(self, dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M:%S')

    def _convert_flight_dt(self, dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M')


class AviaImportSuperkassaMarker(AviaImportMarker):
    """ Import marker from Superkassa """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    def on_execute(self):
        password = self._get_partner_secret('password')
        authhash = self._get_partner_secret('hash')

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=SuperkassaMarkerReader(password, authhash, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
