# -*- coding: utf-8 -*-
import json
from datetime import datetime, timedelta

from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class SvyaznoyMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'http://meta.svyaznoy.travel/multireq/api/stat'
    LOGIN = 'yandex.ru'

    def __init__(self, password, geo_point_cache, logger):
        """
        :type password: str
        """
        super(SvyaznoyMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'notpaid': 'booking',
                'issued': 'paid',
                'cancelled': 'cancel',
                'exchange': 'paid',
            },
            geo_point_cache=geo_point_cache,
        )
        self._password = password

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'date_start': report_date.strftime('%Y-%m-%d'),
                'date_end': report_date.strftime('%Y-%m-%d'),
                'source': self.LOGIN,
                'passwd': self._password,
                'gmt': '0',
            }
        )

    def _get_request_auth(self):
        return None

    def parse_report(self, content):
        orders = []
        try:
            order_list = json.loads(content)
        except ValueError:
            self._logger.exception('Bad SVYAZNOY JSON')
            return []

        for order_dict in order_list:
            (orig, dest) = order_dict.get('direction').split('/')
            status = order_dict.get('status')
            status = status.lower() if isinstance(status, basestring) else status
            order = {
                'order_id': order_dict.get('order_id'),
                'created_at': self._convert_dt(order_dict.get('booking_date')),
                'price': order_dict.get('price'),
                'status': self._parse_status(status),
                'marker': order_dict.get('marker'),
                'airport_from': orig,
                'airport_to': dest,
            }
            orders.append(order)
        return orders

    def _convert_dt(self, dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M:%S')


class AviaImportSvyaznoyMarker(AviaImportMarker):
    """ Import marker from Svyaznoy """

    class Requirements(AviaImportMarker.Requirements):
        pass

    def on_execute(self):
        password = self._get_partner_secret('password')
        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=SvyaznoyMarkerReader(password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
