# -*- coding: utf-8 -*-
from datetime import timedelta, datetime

from sandbox import sdk2
from sandbox.projects.avia.import_marker import AviaImportMarker
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerTransfer, MarkerWriter
from sandbox.sandboxsdk.environments import PipEnvironment


class TinkoffMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://api-travel.tinkoff.ru/api/v1/metasearch/yandex/statistics'

    def __init__(self, geo_point_cache, logger):
        super(TinkoffMarkerReader, self).__init__(
            statuses_map={
                'processing': 'booking',
                'paid': 'paid',
                'cancelled': 'cancel',
            },
            logger=logger,
            geo_point_cache=geo_point_cache,
        )

    def parse_report(self, content):
        for message in content.splitlines()[1:]:
            (origin, destination, created_at, price, currency,
             adults, children, infants, marker, order_id, status) = message.split(',')

            yield {
                'order_id': order_id,
                'created_at': self._convert_dt(created_at),
                'status': self._parse_status(status),
                'marker': marker,
                'airport_from': origin,
                'airport_to': destination,
                'price': float(price),
                'currency': currency
            }

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%d')

    def _get_request_auth(self):
        return None

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'date_from': report_date.strftime('%Y-%m-%d'),
                'date_to': report_date.strftime('%Y-%m-%d'),
            }
        )


class AviaImportTinkoffMarker(AviaImportMarker):
    """ Import marker from Tinkoff """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Import parameters') as import_block:
            partner_code = sdk2.parameters.String('Partner\'s code', required=True, default='tinkoff')
            source = sdk2.parameters.String('Source', required=True, default='tinkoff')

    def on_execute(self):
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=TinkoffMarkerReader(self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
