#! -*- coding: utf-8 -*-
from datetime import datetime
from collections import defaultdict
from io import BytesIO
from itertools import islice

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerWriter
from sandbox.projects.avia.import_marker import AviaImportMarker

# url join?
TRIP_RU_URL = 'http://www.mytrip.com/affiliates'
TRIP_RU_LOGIN_URL = '{trip_ru_url}/Login.aspx'.format(trip_ru_url=TRIP_RU_URL)
TRIP_RU_CLIENT_URL = '{trip_ru_url}/Client/AffiliateCommissions.aspx'.format(trip_ru_url=TRIP_RU_URL)
TRIP_RU_GETEXCEL_URL = '{trip_ru_url}/GetExcel.aspx?kind=Commissions'.format(trip_ru_url=TRIP_RU_URL)


class AviaImportTripRuMarker(AviaImportMarker):
    """ Import marker from TripRu """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('bs4'),
            PipEnvironment('openpyxl'),
        )

    STATUSES_MAP = {
        'Pending': 'booking',
        'Approved': 'paid',
        'Declined': 'cancel',
    }

    def on_execute(self):
        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)

        self._import_trip_ru(self._left_date, self._right_date)

        self._logger.info('Done')

    def get_affiliate_data(self):
        return {
            '496': {
                'login': 'ticket.yandex.ru',
                'password': self._get_partner_secret('ru_password')
            },
            '13429': {
                'login': 'avia.yandex.ua',
                'password': self._get_partner_secret('ua_password')
            },
            '15132': {
                'login': 'avia.yandex.tr',
                'password': self._get_partner_secret('tr_password')
            },
            '19124': {
                'login': 'Yandex_KZ',
                'password': self._get_partner_secret('kz_password')
            },
        }

    def _import_trip_ru(self, left_date, right_date):
        trip_ru_affiliate_data = self.get_affiliate_data()

        for aff_id, aff_data in trip_ru_affiliate_data.iteritems():
            self._logger.info('Get report id=%s', aff_id)

            try:
                excel_report = self._get_affiliate_report(aff_data['login'], aff_data['password'])
                report_rows = self._parse_affiliate_report(excel_report, left_date, right_date, self._partner)
            except Exception:
                self._logger.exception('trip.ru affiliate report import error')
                continue

            rows_by_date = self._group_by_creation_date(report_rows)
            for report_date, rows in rows_by_date.iteritems():
                marker_writer = MarkerWriter(
                    self.Parameters.partner_code,
                    self._logger,
                    self.Parameters.yt_partner_booking_root,
                    self._yt,
                )
                marker_writer.add_rows(rows)
                marker_writer.write_to_yt(report_date)

    def _get_affiliate_report(self, login, password):
        import requests
        from bs4 import BeautifulSoup

        self._logger.info('Get viewstate from: %s', TRIP_RU_LOGIN_URL)
        s = requests.Session()
        r = s.get(TRIP_RU_LOGIN_URL)

        self._logger.info('Parse viewstate')
        soup = BeautifulSoup(r.content)

        view_state = soup.find(id='__VIEWSTATE')['value']
        view_state_generator = soup.find(id='__VIEWSTATEGENERATOR')['value']

        login_data = {
            '__VIEWSTATE': view_state,
            '__VIEWSTATEGENERATOR': view_state_generator,
            'username': login,
            'password': password,
        }

        self._logger.info('Send login form to: %s', TRIP_RU_LOGIN_URL)
        s.post(TRIP_RU_LOGIN_URL, data=login_data)

        self._logger.info('Get client affiliate commissions URL: %s', TRIP_RU_CLIENT_URL)
        s.get(TRIP_RU_CLIENT_URL)

        self._logger.info('Get report file: %s', TRIP_RU_GETEXCEL_URL)
        r = s.get(TRIP_RU_GETEXCEL_URL)

        return r.content

    def _parse_affiliate_report(self, excel_content, left_date, right_date, partner):
        from openpyxl import load_workbook
        from openpyxl.cell.cell import MergedCell

        self._logger.info('Parse excel (%s bytes)', len(excel_content))

        wb = load_workbook(filename=BytesIO(excel_content))
        ws = wb.worksheets[0]

        marker_rows = []
        for row in islice(ws.iter_rows(), 3, None):
            if any(isinstance(c, MergedCell) for c in row):
                continue
            row_values = [c.value for c in row]
            _id, created, _cost, _commission, tr_status, _flight_type, marker = row_values

            if not isinstance(created, datetime):
                continue

            if not left_date <= created.date() <= right_date:
                continue

            status = AviaImportTripRuMarker.STATUSES_MAP.get(tr_status)

            if not status:
                self._logger.error('Unknown status: %s', tr_status)
                continue

            if marker:
                marker_row = MarkerWriter.Row(
                    created_at=created,
                    partner=partner['code'],
                    partner_id=partner['partner_id'],
                    billing_order_id=partner['billing_order_id'],
                    marker=marker,
                    status=status
                )

                marker_rows.append(marker_row)

        return marker_rows

    @staticmethod
    def _group_by_creation_date(table_rows):
        rows_by_date = defaultdict(list)

        for row in table_rows:
            rows_by_date[row.created_at.date()].append(row)

        return rows_by_date
