#! -*- coding: utf-8 -*-
import csv
from datetime import timedelta
from io import BytesIO

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerWriter
from sandbox.projects.avia.import_marker import AviaImportMarker


class AviaImportTripstaMarker(AviaImportMarker):
    """ Import marker from Tripsta """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('bs4'),
            PipEnvironment('pytz'),
            PipEnvironment('python-dateutil'),
        )

    ROOT_URL = 'http://shelly.tripsta.net:8082'
    LOGIN_URL = '{tripsta_url}/login'.format(tripsta_url=ROOT_URL)
    REPORT_URL = '{tripsta_url}/transactions/flight.csv'.format(tripsta_url=ROOT_URL)
    LOGIN = 799855
    STATUSES_MAP = {
        'confirmed': 'paid',
        'cancelled': 'cancel',
    }

    def on_execute(self):
        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        self._import_trip_ru(self._left_date, self._right_date)
        self._logger.info('Done')

    def _import_trip_ru(self, left_date, right_date):
        report_date = left_date
        while report_date <= right_date:
            marker_writer = MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt
            )

            raw_csv = self._get_affiliate_report(report_date)
            marker_writer.add_rows(self._parse_report(raw_csv, self.Parameters.partner_code))
            marker_writer.write_to_yt(report_date)

            report_date += timedelta(days=1)

    def _get_affiliate_report(self, report_date):
        import requests
        from bs4 import BeautifulSoup

        self._logger.info('Get login form: %s', AviaImportTripstaMarker.LOGIN_URL)
        s = requests.Session()
        r = s.get(AviaImportTripstaMarker.LOGIN_URL)

        self._logger.info('Parse authenticity_token')
        soup = BeautifulSoup(r.content)
        authenticity_token = soup.find('input', {'name': 'authenticity_token'})['value']

        login_data = {
            'partner_login[code]': AviaImportTripstaMarker.LOGIN,
            'partner_login[password]': self._get_partner_secret('password'),
            'authenticity_token': authenticity_token,
        }

        self._logger.info('Send login form to: %s', AviaImportTripstaMarker.LOGIN_URL)
        s.post(AviaImportTripstaMarker.LOGIN_URL, data=login_data)

        report_data = {
            'commit': 'Search',
            'market': 'ru',
            'partner': AviaImportTripstaMarker.LOGIN,
            'reservation_date_from': report_date.strftime('%Y-%m-%d'),
            'reservation_date_to': report_date.strftime('%Y-%m-%d'),
            'state': '',
            'type': 'flight',
        }

        self._logger.info('Get client affiliate commissions URL: %s', AviaImportTripstaMarker.REPORT_URL)
        r = s.get(AviaImportTripstaMarker.REPORT_URL, params=report_data)

        return r.content

    @staticmethod
    def _utc_to_msk(utc_dt):
        import pytz

        moscow_timezone = pytz.timezone('Europe/Moscow')
        local_dt = utc_dt.replace(tzinfo=pytz.utc).astimezone(moscow_timezone)
        return moscow_timezone.normalize(local_dt)

    @staticmethod
    def _parse_report(raw_csv, partner_code):
        from dateutil.parser import parse

        reader = csv.reader(BytesIO(raw_csv), delimiter=',')
        reader.next()
        rows = []
        for row in reader:
            """
            #,company_name,reservation_time,update_time,checkout_date,state,type,id,opt_value,...
            """
            reservation_time = AviaImportTripstaMarker._utc_to_msk(parse(row[2]))
            update_time = AviaImportTripstaMarker._utc_to_msk(parse(row[3]))

            state = row[5]
            marker = row[8]
            if marker:
                row = MarkerWriter.Row(
                    partner=partner_code,
                    marker=marker,
                    status=AviaImportTripstaMarker.STATUSES_MAP.get(state),
                    created_at=reservation_time,
                    confirm_dt=update_time,
                )

                rows.append(row)

        return rows
