# -*- coding: utf-8 -*-
from datetime import timedelta, datetime

from sandbox import sdk2
from sandbox.projects.avia.import_marker import AviaImportMarker
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerTransfer, MarkerWriter
from sandbox.sandboxsdk.environments import PipEnvironment


class TutuMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://partners-proxy.production.avia.yandex.net/tutu/yapi/orders/'

    def __init__(self, geo_point_cache, logger):
        super(TutuMarkerReader, self).__init__(
            statuses_map={
                'booked': 'booking',
                'paid': 'paid',
                'cancel': 'cancel',
                'refunded': 'cancel',
            },
            logger=logger,
            geo_point_cache=geo_point_cache,
        )

    def parse_report(self, content):
        import ujson

        orders = []
        try:
            response = ujson.loads(content)
            bookings = response['bookings']['booking']
        except ValueError:
            self._logger.exception('Bad Tutu JSON')
            return []

        for booking in bookings:
            try:
                flights = [
                    {
                        'from': flight.get('departure'),
                        'to': flight.get('arrival'),
                        'departure_dt': datetime.strptime(
                            '{} {}'.format(flight.get('departureDate'), flight.get('departureTime')),
                            '%Y-%m-%d %H:%M',
                        ),
                        'arrival_dt': datetime.strptime(
                            '{} {}'.format(flight.get('arrivalDate'), flight.get('arrivalTime')),
                            '%Y-%m-%d %H:%M',
                        ),
                    }
                    for flight in booking['segment']['flight']
                ] if booking.get('segment') else []
                order = {
                    'order_id': booking.get('id'),
                    'created_at': self._convert_dt(booking.get('created_at')),
                    'price': float(booking.get('price')),
                    'currency': booking.get('currency'),
                    'status': self._parse_status(booking.get('status')),
                    'marker': booking.get('marker'),
                    'flights': flights,
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        self._fillin_airports(orders)
        return orders

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%d')

    def _get_request_auth(self):
        return None

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'dateFrom': report_date.strftime('%Y-%m-%d'),
                'dateTo': report_date.strftime('%Y-%m-%d'),
            }
        )


class AviaImportTutuMarker(AviaImportMarker):
    """ Import marker from Tutu """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('ujson'),
        )

    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Import parameters') as import_block:
            partner_code = sdk2.parameters.String('Partner\'s code', required=True, default='tutu')
            source = sdk2.parameters.String('Source', required=True, default='tutu')

    def on_execute(self):
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=TutuMarkerReader(self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
