# -*- coding: utf-8 -*-
import json
from datetime import datetime

from sandbox.projects.avia.lib.marker import BatchMarkerReader, MarkerWriter, MarkerBatchTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class UfsMarkerReader(BatchMarkerReader):
    REQUEST_URL = 'https://ym.ufs-online.ru/webservices/avia/Metasearch/OrderList/yandex'
    LOGIN = 'YandexOOO'

    def __init__(self, password, geo_point_cache, logger):
        """
        :param str password: пароль для доступа
        """
        super(UfsMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'Paid': 'paid',
                'Cancelled': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._password = password

    def _get_request_data(self):
        return (
            self.REQUEST_URL,
            {},
        )

    def _get_request_auth(self):
        return self.LOGIN, self._password

    def parse_report(self, content):
        orders = []
        try:
            raw_report = json.loads(content)
        except ValueError:
            self._logger.error('Bad JSON from partner')
            return orders

        for booking in raw_report:
            try:
                order = {
                    'order_id': booking.get('OrderId'),
                    'created_at': self._convert_dt(booking.get('CreatDateTime')),
                    'price': booking.get('Amount'),
                    'currency': booking.get('Currency'),
                    'from': booking.get('From'),
                    'to': booking.get('To'),
                    'status': self._parse_status(booking.get('Status')),
                    'marker': booking.get('Marker'),
                    'trip_type': self._parse_trip_type(booking.get('TripType')),
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')

        return orders

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%d.%m.%Y %H:%M:%S')

    def _parse_trip_type(self, trip_type):
        if trip_type == 'RoundTrip':
            return 'roundtrip'

        if trip_type == 'OneWay':
            return 'oneway'

        self._logger.info('Unknown trip_type: %s', trip_type)
        return trip_type


class AviaImportUfsMarker(AviaImportMarker):
    """ Import marker from UfsPlane """

    class Requirements(AviaImportMarker.Requirements):
        pass

    def on_execute(self):
        password = self._get_partner_secret('password')

        reader = UfsMarkerReader(password, self.geo_point_cache, self._logger)
        marker_transfer = MarkerBatchTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=reader,
            logger=self._logger,
        )

        marker_transfer.transfer()
