# coding=utf-8
from __future__ import unicode_literals

from collections import namedtuple
from datetime import datetime, timedelta
from hashlib import md5

from sandbox import sdk2
from sandbox.projects.avia.import_marker import AviaImportMarker
from sandbox.projects.avia.lib.marker import BaseMarkerReader, MarkerTransfer, MarkerWriter
from sandbox.sandboxsdk.environments import PipEnvironment

REQUEST_URL = 'https://partners-proxy.production.avia.yandex.net/utair/bitrix/components/travelshop/ibe.rest/GetOrderList/'
TIMEOUT = 30
NUMBER_OF_ATTEMPTS = 3
WAIT_BETWEEN_ATTEMPTS_MS = 60 * 1000

SecretKeys = namedtuple('SecretKeys', 'login_key, password_key, shared_secret_key')

PARTNER_SECRET_ID = 'sec-01e4nxqyyqzpfjv2y0b82v3b68'
SECRET_KEYS = SecretKeys('login', 'password', 'shared-secret')
MOBILE_SECRET_KEYS = SecretKeys('mobile-report-api-login', 'mobile-report-api-password',
                                'mobile-report-api-shared-secret')


class UtairMarkerReader(BaseMarkerReader):
    def __init__(self, login, password, shared_secret, logger, request_timeout, geo_point_cache):
        super(UtairMarkerReader, self).__init__(
            statuses_map={
                'S': 'booking',
                'B': 'booking',
                'X': 'cancel',
                'T': 'paid',
            },
            logger=logger,
            geo_point_cache=geo_point_cache,
        )
        self._login = login
        self._password = password
        self._shared_secret = shared_secret
        self._trip_type_cache = {
            'RT': 'roundtrip',
            'OW': 'oneway',
            'OJ': 'openjaw',
        }
        self._request_timeout = request_timeout

    def import_data(self, date):
        import requests
        from retrying import retry

        @retry(stop_max_attempt_number=NUMBER_OF_ATTEMPTS, wait_fixed=WAIT_BETWEEN_ATTEMPTS_MS)
        def get_data(url, params):
            self._logger.info('Get %s', url)
            return requests.post(url, json=params, timeout=self._request_timeout)

        self._logger.info('Process: %s', date.strftime('%Y-%m-%d'))
        params = self._get_request_data(date)
        response = get_data(REQUEST_URL, params)
        response.raise_for_status()
        return list(
            filter(
                lambda item: item.get('created_at').date() == date,
                self.parse_report(response.json()),
            )
        )

    def _get_request_data(self, date):
        from_date = date - timedelta(days=1)
        to_date = date + timedelta(days=1)

        from_date = from_date.strftime('%d.%m.%Y')
        to_date = to_date.strftime('%d.%m.%Y')

        self._logger.info('%s - %s', from_date, to_date)

        session_token = ':'.join([self._login, self._password, 'N'])
        hash = self._get_hash(session_token, from_date, to_date)

        return dict(
            session_token=session_token,
            timestamp_from=from_date,
            timestamp_to=to_date,
            hash=hash,
        )

    def _get_hash(self, session_token, from_date, to_date):
        m = md5()
        m.update((session_token + from_date + to_date + self._shared_secret).encode('utf-8'))
        return m.hexdigest()

    def parse_report(self, data):
        self._ensure_response(data)

        if not data['orders']:
            self._logger.info('Order list is empty')
            return

        currency = data['currency']
        for order in data['orders']:
            try:
                flights = [
                    self._parse_flight(flight_dict)
                    for flight_dict
                    in order.get('segments', [])
                ]

                yield {
                    'order_id': order['order_id'],
                    'created_at': self._convert_dt(order.get('booking_timestamp')),
                    'price': order.get('full_price'),
                    'currency': currency,
                    'status': self._parse_status(order.get('status')),
                    'marker': order.get('partner_data'),
                    'flights': flights,
                    'airport_from': order['departure_point'],
                    'airport_to': order['arrival_point'],
                    'trip_type': self._get_trip_type(order['owrt']),
                }
            except Exception:
                self._logger.exception('Parse error. Order:\n%s', order)
                raise

    @staticmethod
    def _ensure_response(data):
        if 'error' not in data:
            raise ValueError('Incorrect data %s' % data)
        error = data['error']
        if error['code'] != 'OK':
            raise ValueError('Bad response: code=%s, desc=%s' % (error['code'], error['description']))

    def _parse_flight(self, segment):
        try:
            return {
                'from': segment['departure_airport_code'],
                'to': segment['arrival_airport_code'],
                'departure_dt': self._convert_dt(segment['departure_utc']),
                'arrival_dt': self._convert_dt(segment['arrival_utc']),
            }
        except Exception:
            self._logger.exception(
                'Error parse flights for segment %s',
                repr(segment)
            )
            raise

    @staticmethod
    def _convert_dt(dt_str):
        if not dt_str:
            return dt_str
        return datetime.strptime(dt_str, '%d.%m.%Y %H:%M:%S')

    def _get_trip_type(self, code):
        if code in self._trip_type_cache:
            return self._trip_type_cache[code]
        self._logger.warn('Unexpected trip type %s', code)


class AviaImportUtairMarker(AviaImportMarker):
    """ Import marker from Utair """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('requests'),
            PipEnvironment('retrying'),
        )

    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Import type') as import_type_group:
            with sdk2.parameters.RadioGroup('Mode', required=True) as mode:
                mode.values['desktop'] = mode.Value('desktop', default=True)
                mode.values['mobile'] = mode.Value('mobile')

        request_timeout = sdk2.parameters.Integer('Request timeout', required=True, default=TIMEOUT)

    def on_execute(self):
        secret_data = sdk2.yav.Secret(PARTNER_SECRET_ID).data()
        secret_keys = MOBILE_SECRET_KEYS if self.Parameters.mode == 'mobile' else SECRET_KEYS
        login = secret_data[secret_keys.login_key]
        password = secret_data[secret_keys.password_key]
        shared_secret = secret_data[secret_keys.shared_secret_key]

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=UtairMarkerReader(
                login,
                password,
                shared_secret,
                self._logger,
                self.Parameters.request_timeout,
                self.geo_point_cache,
            ),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
