# -*- coding: utf-8 -*-
import calendar
import json
import os
from abc import ABCMeta, abstractmethod
from datetime import datetime, timedelta
from itertools import groupby

import requests
import requests.auth
from requests.adapters import HTTPAdapter
from six.moves import zip

from sandbox.projects.avia.lib.logs import remove_secrets


class MarkerWriter(object):
    class Row(object):
        class BadStatusChoice(Exception):
            pass

        class BadTripTypeChoice(Exception):
            pass

        def __init__(
            self,
            partner,
            marker,
            status,
            created_at,
            confirm_dt=None,
            partner_id=None,
            billing_order_id=None,
            order_id=None,
            price=None,
            currency=None,
            airport_from=None,
            airport_to=None,
            trip_type=None,
            ticket_number=None,
        ):
            self._status_choices = {'booking', 'paid', 'cancel'}
            self._trip_type_choices = {'oneway', 'roundtrip', 'openjaw', None}

            if not isinstance(created_at, datetime):
                raise TypeError('created_at=%r' % created_at)

            if confirm_dt and not isinstance(confirm_dt, datetime):
                raise TypeError('confirm_dt=%r' % confirm_dt)

            if status not in self._status_choices:
                raise self.BadStatusChoice(repr(status))

            if trip_type not in self._trip_type_choices:
                raise self.BadTripTypeChoice(repr(trip_type))

            self.created_at = created_at
            self.confirm_dt = confirm_dt
            self.marker = marker or ''
            self.status = status
            self.partner = partner
            self.partner_id = int(partner_id)
            self.billing_order_id = int(billing_order_id)
            self.order_id = order_id
            self.price = float(price) if price is not None else None
            self.currency = currency
            self.airport_from = airport_from
            self.airport_to = airport_to
            self.trip_type = trip_type
            self.ticket_number = ticket_number

        def __repr__(self):
            return 'Row object [%s]' % vars(self)

    def __init__(self, source, logger, yt_root, yt):
        self._yt_root = yt_root
        self._rows = []
        self._logger = logger
        self._unixtime = calendar.timegm(datetime.utcnow().timetuple())
        self.source = source
        self.yt = yt

    def add_rows(self, rows):
        self._rows.extend(rows)

    @property
    def _json_rows(self):
        for row in self._rows:
            yt_row = {
                'unixtime': self._unixtime,
                'partner': row.partner,
                'marker': row.marker,
                'status': row.status,
                'created_at': self._json_dt(row.created_at),
                'confirm_dt': self._json_dt(row.confirm_dt),
                'source': self.source,
                'partner_id': row.partner_id,
                'billing_order_id': row.billing_order_id,
                'order_id': row.order_id,
                'price': row.price,
                'currency': row.currency,
                'from': row.airport_from,
                'to': row.airport_to,
                'trip_type': row.trip_type,
                'ticket_number': row.ticket_number,
            }

            yield yt_row

    @staticmethod
    def _json_dt(dt):
        return dt.strftime('%Y-%m-%d %H:%M:%S') if dt else None

    def write_to_yt(self, report_date):
        yt_table = self.yt.TablePath(
            os.path.join(self._yt_root, report_date.strftime('%Y-%m-%d')),
            append=True,
        )

        if not self._rows:
            self._logger.info('Nothing to write to: %s', yt_table)
            return

        if not self.yt.exists(self._yt_root):
            self._logger.info('Creating cypress for %s', self._yt_root)
            self.yt.create('map_node', self._yt_root)

        if not self.yt.exists(yt_table):
            self._logger.info('Creating table %s', yt_table)
            self.yt.create_table(
                yt_table,
                attributes={
                    'schema': [
                        {
                            'name': 'unixtime',
                            'type': 'int64',
                            'required': True,
                        },
                        {
                            'name': 'source',
                            'type': 'string',
                        },
                        {
                            'name': 'partner',
                            'type': 'string',
                            'required': True,
                        },
                        {
                            'name': 'partner_id',
                            'type': 'int64',
                            'required': True,
                        },
                        {
                            'name': 'billing_order_id',
                            'type': 'int64',
                            'required': True,
                        },
                        {
                            'name': 'marker',
                            'type': 'string',
                            'required': True,
                        },
                        {
                            'name': 'status',
                            'type': 'string',
                            'required': True,
                        },
                        {
                            'name': 'created_at',
                            'type': 'string',
                            'required': True,
                        },
                        {
                            'name': 'confirm_dt',
                            'type': 'string',
                        },
                        {
                            'name': 'order_id',
                            'type': 'string',
                        },
                        {
                            'name': 'price',
                            'type': 'double',
                        },
                        {
                            'name': 'currency',
                            'type': 'string',
                        },
                        {
                            'name': 'from',
                            'type': 'string',
                        },
                        {
                            'name': 'to',
                            'type': 'string',
                        },
                        {
                            'name': 'trip_type',
                            'type': 'string',
                        },
                        {
                            'name': 'ticket_number',
                            'type': 'string',
                        },
                    ],
                },
            )

        self._flush_rows(yt_table)

    @staticmethod
    def _create_key(row):
        return (
            row['created_at'],
            row['marker'] or 'unknown',
            row['status'] or 'unknown',
            row['order_id'] or 'unknown',
            row['source'],
            row['partner'],
            row.get('ticket_number') or 'unknown',
        )

    def _filter_duplicated_rows(self, yt_table, rows):
        import yt.wrapper as yt_wrapper

        if self.yt.exists(yt_table):
            uniq_keys = {
                self._create_key(r) for r in self.yt.read_table(yt_table, format=yt_wrapper.format.JsonFormat())
            }
        else:
            uniq_keys = set()

        self._logger.info('Start: filter duplicate rows. New rows: %d', len(rows))

        temp = []
        for row in rows:
            key = self._create_key(row)
            if key not in uniq_keys:
                temp.append(row)
                uniq_keys.add(key)

        self._logger.info('Finish: filter duplicate rows. Uniq new rows: %d', len(temp))

        return temp

    def _filter_broken_rows(self, rows):
        self._logger.info('Start: filter broken rows. Uniq new rows: %d', len(rows))
        temp = []
        for r in rows:
            if not all(field for field in self._create_key(r)):
                self._logger.info('Filter row %r, because some required field is empty', r)
                continue
            temp.append(r)

        self._logger.info('Finish: filter broken rows. Uniq new rows: %d', len(temp))

        return temp

    def _flush_rows(self, yt_table):
        import yt.wrapper as yt_wrapper

        self._logger.info('Write YT table: %s', yt_table)

        rows = list(self._json_rows)
        rows = self._filter_duplicated_rows(yt_table, rows)
        rows = self._filter_broken_rows(rows)

        self.yt.write_table(yt_table, rows, format=yt_wrapper.YsonFormat())
        self._rows = []


class DummyWriter(MarkerWriter):
    def write_to_yt(self, date):
        self._logger.info('Orders received for date %s : %s', date, self._rows)
        self._rows = []


class BaseMarkerTransfer(object):
    __metaclass__ = ABCMeta

    def __init__(self, partner, marker_writer, marker_reader, logger):
        """
        :type partner: dict
        :type marker_writer: sandbox.projects.avia.lib.marker.MarkerWriter
        :type marker_reader: sandbox.projects.avia.lib.marker.BaseMarkerReader
        :type logger: logging.Logger
        """
        self._partner = partner
        self._marker_writer = marker_writer
        self._marker_reader = marker_reader
        self._logger = logger

    def export_data(self, orders, date):
        """
        :type orders: list[dict[str,Any]]
        :param orders: список заказов. Формат смотри в BaseMarkerReader.parse_report
        :type date: datetime
        :param date: дата на которую нужно записать данные в YT
        """
        rows = []
        for order in orders:
            try:
                row = MarkerWriter.Row(
                    partner=self._partner['code'],
                    partner_id=self._partner['partner_id'],
                    billing_order_id=self._partner['billing_order_id'],
                    order_id=order.get('order_id'),
                    created_at=order.get('created_at'),
                    price=order['price'],
                    currency=order.get('currency'),
                    airport_from=order.get('airport_from'),
                    airport_to=order.get('airport_to'),
                    status=order['status'],
                    marker=order['marker'],
                    confirm_dt=order.get('confirm_date'),
                    trip_type=order.get('trip_type'),
                    ticket_number=order.get('ticket_number'),
                    # confirm_dt - кандидат на удаление
                    # (смотри: https://st.yandex-team.ru/RASPTICKETS-13086#1536754443000)
                )
                rows.append(row)
            except Exception:
                self._logger.exception('Problems with order %s', order)

        self._marker_writer.add_rows(rows)
        self._marker_writer.write_to_yt(date)
        self._logger.info('%d records parsed for date %s', len(rows), date)


class MarkerTransfer(BaseMarkerTransfer):
    def transfer(self, date):
        """
        Принимаем данные из marker_reader и передаем в marker_writer
        :type date: datetime
        :param date: дата, на которую нужно брать данные у партнера и перекладывать в YT
        """
        import six
        import sys

        try:
            orders = self._marker_reader.import_data(date)
            self.export_data(orders, date)
        except Exception:
            self._logger.exception(
                'Error transfering data for partner %s, date %s',
                self._partner,
                date,
            )
            exc_info = sys.exc_info()
            six.reraise(*exc_info)
        else:
            self._logger.info('Data transfer complete')


class BaseMarkerReader(object):
    __metaclass__ = ABCMeta

    def __init__(self, statuses_map, logger, geo_point_cache):
        """
        :type logger: logging.Logger
        :param logger: logging.Logger логгер для проброса сообщений и ошибок
        :type statuses_map: dict[str, str]
        :param statuses_map: словарь преобразования статусов от партнера в наши
        :type: geo_point_cache: GeoPointCache
        :param geo_point_cache: Кэш сопоставления iata/sirena -> city_id
        """
        self._logger = logger
        self._statuses_map = statuses_map
        self.geo_point_cache = geo_point_cache

    @abstractmethod
    def import_data(self, date):
        """
        Собираем данные от партнера и сохраняем в промежуточную структуру
        Формат стурктуры смотри в parse_report

        :type date: datetime
        :param date: дата, на которую нужно забрать данные
        :rtype: list[dict[str, Any]]
        :return: формат возвращаемой структуры смотри в parse_report
        """

    @abstractmethod
    def parse_report(self, content):
        """
        Каждый подкласс должен переопределять этот метод и возвращать
        массив заказов со следующей структурой
        [{
            order_id: ...,

            created_at: ...,  --datetime format

            confirm_date: ...,  --datetime format

            price: ...,

            currency: ...,

            status: ...,  (booking, paid, cancel)

            marker: ...,

            airport_from: ...,  *** IATA

            airport_to: ...,  *** IATA

            trip_type: ..., 'oneway' | 'roundtrip' | 'openjaw' ***

        }, ...]

        Поля From, To и TripType (помеченные звездочкой) могут быть автоматически
        заполнены. Для автоматического заполнения нужно добавить по ключу flights список аэопортов,
        через которые проходил маршрут, а также даты перелетов и вызвать fillin_airports.
        Подробнее структура, которую нужно заполнить описана в get_airports.
        Вызывать fillin_airport каждый наследник должен сам внутри parse_report

        :type content: bytes
        :param content: сырые данные от поставщика
        :rtype: list[dict[str, Any]]
        """

    def _parse_status(self, status):
        """
        Преобразуем строчку статуса, которую нам передает партнер в наш статус
        При создании экземпляра класса нужно внимательно следить за statuses_map.
        На данном этапе соответствие статусам в YT не проверяется, поэтому
        оно может упасть позже, когда этот статус будет передан MarkerWriter

        :type status: str
        :param status: Статус от партнера
        :rtype: str
        :return: наш статус для YT (см lib.marker.MarkerWriter.Row)
        """
        parsed = self._statuses_map.get(status)
        if status and not parsed:
            raise Exception('Unknown status: {}'.format(status))
        return parsed

    def _fillin_airports(self, orders):
        """
        Принимаем словарик заказов и заполняем начальную и конечную точку маршрута.
        Информацию о структуре входных данных смотрим в _get_airports

        :type orders: list[dict[str, Any]]
        :param orders: список заказов в виде словарей.
        """
        for order in orders:
            (From, To), trip_type = self._get_airports(order)
            order['airport_from'] = From
            order['airport_to'] = To
            order['trip_type'] = trip_type
        return None

    def _get_airports(self, order):
        """
        Получаем из неупорядоченного списка сегментов пункт начала маршрута и пункт конца.
        Проверяем, что это маршрут закольцованный и если так, то ищем середину
        Проверяем, что это один цельный маршрут.

        Формат структуры по ключу flights:
        [{
            from: ..., -- IATA код аэропорта отлета

            to: ...,  -- IATA код аэропорта прилета

            departure_dt: ...,  --дата и время отлета datetime

            arrival_dt: ...,  --дата и время прилета datetime
        }]

        :type order: dict[basestring, list[dict[basestring, Any]]]
        :param order: Заказ, в котором по ключу flights лежит список полетов
        :rtype: ((basestring, basestring), basestring)
        :return: ((начальная точка (IATA), конечная точка (IATA)), тип поездки)
        """
        if not order.get('flights'):
            self._logger.info('No segments found in order %s', order)
            return (None, None), None

        flights = sorted(order['flights'], key=lambda flight: flight['departure_dt'])

        orig = flights[0]['from']
        dest = flights[-1]['to']

        try:
            if self._is_round_trip(flights):
                dest = self._most_delayed_airport(flights)
                trip_type = 'roundtrip'
            else:
                trip_type = 'oneway'

            if not self._is_continuous(flights):
                trip_type = 'openjaw'
        except KeyError:
            trip_type = None

        return (orig, dest), trip_type

    def _is_round_trip(self, flights):
        """
        Определяем, прилетели ли мы в тот же город, из которого улетали

        :type flights: list[dict[str,Any]]
        :param flights: Список словарей с ключами from и to и значениями в виде IATA кодов
        :rtype: bool
        :return: True если вернулись откуда прилетели, иначе False
        """
        if not flights or len(flights) == 1:
            return False
        try:
            orig = flights[0]['from']
            dest = flights[-1]['to']
            if orig is None or dest is None:
                return False
            if orig == dest or self._settlement_id(orig) == self._settlement_id(dest):
                return True
            return False
        except KeyError:
            self._logger.warning('Problem detecting round trip in flight %s', str(flights))
            raise

    def _is_continuous(self, flights):
        """
        Определяем, является ли маршрут географически непрерывным,
        то есть нет прилета в один город, а вылета из другого

        :type flights: list[dict[str, Any]]
        :param flights: список словарей полетов по маршруту с ключами from и to.
        Значения ключей from и to - IATA коды аэропорта
        :return: True если полет нерперывный, иначе False
        """
        if not flights:
            raise Exception('No flights found')
        if len(flights) == 1:
            return True
        try:
            for (past, future) in zip(flights, flights[1:]):
                inport = past['to']
                outport = future['from']
                if inport is None or outport is None:
                    self._logger.info('Unknown airport in %s', str(flights))
                    return False

                if inport != outport and self._settlement_id(inport) != self._settlement_id(outport):
                    return False
        except KeyError:
            self._logger.exception('Problems detecting flight continuity: %s', str(flights))
            raise

        return True

    @staticmethod
    def _most_delayed_airport(flights):
        """
        Находим среди всех перелетов аэропорт, в котором находились дольше всего
        Необходимо при поиске направления в случае round trip

        :type flights: list[dict[str, Any]]
        :param flights: список словарей с ключами from (IATA), to (IATA), deprature_dt (datetime),
        arrival_dt (datetime)
        :rtype: str
        :return: Аэропорт, в котором была наибольшая задержка между перелетами
        """
        if not flights:
            raise Exception('No flights found')
        if len(flights) == 1:
            raise Exception('Cannot determine for a single flight')

        max_delay = timedelta(0)
        max_delay_airport = None
        for (past, future) in zip(flights, flights[1:]):
            delay = future['departure_dt'] - past['arrival_dt']
            if delay > max_delay:
                max_delay = delay
                if future['from'] is not None:
                    max_delay_airport = future['from']
                else:
                    max_delay_airport = past['to']
        return max_delay_airport

    def _settlement_id(self, iata):
        try:
            return self.geo_point_cache.settlement_id_by_code(iata.encode('utf-8'))
        except KeyError:
            self._logger.exception('Problems with airport %s', str(iata))
            raise


def get_session(retries=3):
    session = requests.Session()
    adapter = HTTPAdapter(max_retries=retries)
    session.mount('http://', adapter=adapter)
    session.mount('https://', adapter=adapter)
    return session


class MarkerHTTPReader(BaseMarkerReader):
    __metaclass__ = ABCMeta

    def __init__(self, statuses_map, logger, geo_point_cache, timeout=(10.0, 10.0), retries=10, ssl_verify=True):
        """
        :type logger: logging.Logger
        :param logger: loggin.Logger логгер для проброса сообщений и ошибок
        :type statuses_map: dict[str, str]
        :param statuses_map: словарь преобразования статусов от партнера в наши
        :type timeout: Optional[tuple[float, float]]
        """
        super(MarkerHTTPReader, self).__init__(
            logger=logger,
            statuses_map=statuses_map,
            geo_point_cache=geo_point_cache,
        )
        self._session = get_session(retries=retries)
        self.timeout = timeout
        self.ssl_verify = ssl_verify

    def import_data(self, date):
        """
        Собираем данные от партнера и сохраняем в промежуточную структуру
        Формат структуры смотри в parse_report

        :type date: datetime
        :param date: дата, на которую нужно забрать данные
        :rtype: list[dict[str, Any]]
        :return: формат возвращаемой структуры смотри в parse_report
        """
        self._logger.info('Process: %s', date.strftime('%Y-%m-%d'))
        (url, params) = self._get_request_data(date)
        self._logger.info('Get %s, %s', url, remove_secrets(params))
        r = self._session.get(url, auth=self._get_request_auth(), params=params, timeout=self.timeout, verify=self.ssl_verify)
        return self.parse_report(r.content)

    @abstractmethod
    def _get_request_auth(self):
        """
        :rtype: requests.auth.AuthBase
        :return: тип аутентификации у партнера, например
        requests.auth.HTTPBasicAuth
        """

    @abstractmethod
    def _get_request_data(self, report_date):
        """
        Возвращает кортеж идентифицирующий место, откуда брать данные для
        текущей даты

        :type report_date: datetime
        :param report_date: дата на которую нужены данные
        :rtype: (str, dict[str, str])
        :return: (url, params) \
        url - адрес, по которому нужно запрашивать данные \
        params - словарь параметров, которые нужно передавать по url
        """


class JSONRPCMarkerReader(MarkerHTTPReader):
    """Marker Reader for ticketsru and bookandtrip"""

    def __init__(self, request_url, user_key, user_token, user_name, user_password, geo_point_cache, logger):
        super(JSONRPCMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'W': 'booking',
                'NP': 'booking',
                'NC': 'booking',
                'CM': 'booking',
                'WA': 'booking',
                'P': 'paid',
                'PN': 'paid',
                'CR': 'paid',
                'R': 'paid',
                'OR': 'paid',
                'LCP': 'paid',
                'PC': 'cancel',
                'C': 'cancel',
                'CO': 'cancel',
                'CU': 'cancel',
                'CL': 'cancel',
                'WD': 'cancel',
                'Cancelled': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._user_token = user_token
        self._user_key = user_key
        self._user_name = user_name
        self._user_password = user_password
        self._request_url = request_url

    def _get_request_data(self, report_date):
        return (
            self._request_url,
            {
                'key': self._user_key,
                'user_token': self._user_token,
                'date_from': (report_date - timedelta(days=1)).strftime('%d.%m.%Y'),
                'date_to': (report_date + timedelta(days=1)).strftime('%d.%m.%Y'),
                'gmt': '0',
            },
        )

    def _get_request_auth(self):
        return self._user_name, self._user_password

    def import_data(self, date):
        """
        Собираем данные от партнера и сохраняем в промежуточную структуру
        Формат стурктуры смотри в parse_report

        :param datetime date: дата, на которую нужно забрать данные
        :rtype: list[dict[str, Any]]
        :return: формат возвращаемой структуры смотри в parse_report
        """
        self._logger.info('Process: %s', date.strftime('%Y-%m-%d'))
        (url, params) = self._get_request_data(date)
        self._logger.info('Get %s, %s', url, remove_secrets(params))
        r = requests.post(
            url,
            auth=self._get_request_auth(),
            params=params,
        )
        orders = self.parse_report(r.content)
        for order in orders:
            if order['created_at'].date() == date:
                yield order

    def parse_report(self, content):
        orders = []
        try:
            json_rpc_data = json.loads(content)
            if 'json' not in json_rpc_data:
                raise ValueError('Incorrect jsonrpc data %s' % json_rpc_data)
            if not '200' == json_rpc_data['json'].get('status'):
                raise ValueError('Bad jsonrpc status: %s' % json_rpc_data['json'].get('status'))
            raw_data = json_rpc_data['json'].get('result')
            order_list = json.loads(raw_data)
        except ValueError:
            self._logger.exception('Bad TICKETS RU JSON %s', content)
            return []

        for order_dict in order_list['items']:
            try:
                flight_dicts = [
                    {
                        'from': flight.get('departure_location'),
                        'to': flight.get('arrival_location'),
                        'departure_dt': self._convert_dt(flight.get('departure_time')),
                        'arrival_dt': self._convert_dt(flight.get('arrival_time')),
                    }
                    for flight in order_dict['locations']
                ]
                order = {
                    'order_id': order_dict.get('locator'),
                    'created_at': self._convert_dt(order_dict.get('date')),
                    'price': order_dict['final_all_price']['RUR'],
                    'currency': 'RUR',
                    'status': self._parse_status(order_dict.get('status')),
                    'marker': order_dict.get('marker'),
                    'flights': flight_dicts,
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        self._fillin_airports(orders)
        return orders

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.fromtimestamp(dt_str)


class BatchMarkerReader(BaseMarkerReader):
    __metaclass__ = ABCMeta

    def __init__(self, statuses_map, logger, geo_point_cache):
        """
        :type logger: logging.Logger
        :param logger: loggin.Logger логгер для проброса сообщений и ошибок
        :type statuses_map: dict[str, str]
        :param statuses_map: словарь преобразования статусов от партнера в наши
        """
        super(BatchMarkerReader, self).__init__(
            logger=logger,
            statuses_map=statuses_map,
            geo_point_cache=geo_point_cache,
        )

    def import_data(self):
        """
        Собираем данные от партнера и сохраняем в промежуточную структуру
        Формат стурктуры смотри в parse_report

        :rtype: list[dict[str, Any]]
        :return: формат возвращаемой структуры смотри в parse_report
        """
        self._logger.info('Importing data')
        (url, params) = self._get_request_data()
        self._logger.info('Get %s, %s', url, remove_secrets(params))
        r = requests.get(
            url,
            auth=self._get_request_auth(),
            params=params,
        )
        self._logger.info('Parsing report')
        return self.parse_report(r.content)

    @abstractmethod
    def _get_request_data(self):
        """
        Возвращает кортеж идентифицирующий место откуда брать данные

        :type report_date: datetime
        :param report_date: дата на которую нужены данные
        :rtype: (str, dict[str, str])
        :return: (url, params) \
        url - адрес, по которому нужно запрашивать данные \
        params - словарь параметров, которые нужно передавать по url
        """


class MarkerBatchTransfer(BaseMarkerTransfer):
    def __init__(self, partner, marker_writer, marker_reader, logger):
        super(MarkerBatchTransfer, self).__init__(
            partner=partner,
            marker_writer=marker_writer,
            marker_reader=marker_reader,
            logger=logger,
        )

    def transfer(self):
        """
        Принимаем все возможные данные из marker_reader и передаем в marker_writer
        """
        import six
        import sys

        try:
            orders = list(sorted(self._marker_reader.import_data(), key=lambda x: x['created_at']))
            self._logger.info('%d orders will be transfered', len(orders))
            for date, date_orders in groupby(orders, lambda x: x['created_at'].date()):
                self._logger.info('Transfering group for date %s', date)
                self.export_data(date_orders, date)
        except ImportError:
            exc_info = sys.exc_info()
            six.reraise(*exc_info)
        except Exception:
            self._logger.exception(
                'Error transfering data for partner %s',
                self._partner,
            )
            raise
        else:
            self._logger.info('Data transfer complete')
