# coding: utf-8
import logging

from six import iteritems

from sandbox.projects.avia.lib import yt_helpers

logging = logging.getLogger(__name__)


class SchemaComparisonResult(object):
    Incompatible = -1
    RightGreater = 1
    LeftGreater = 2
    BothGreater = 3
    Equal = 0


class TableMergeError(Exception):
    pass


def merge_table(ytw, table, dst_table):
    if not ytw.exists(dst_table):
        _create_table(ytw, dst_table, image=table)
        ytw.set_attribute(dst_table, 'processed', [])
    else:
        if not ytw.exists(dst_table + '/@processed'):
            #  При автоматическом мерже создается и обновляется атрибут processed.
            #  Если его нет, это признак проблем
            raise TableMergeError('Table {} exists, but does not have [processed] attribute'.format(dst_table))

    processed = ytw.get_attribute(dst_table, 'processed')
    if table in processed:
        return

    with ytw.Transaction():
        schema = ytw.get_attribute(table, 'schema')
        destination_schema = ytw.get_attribute(dst_table, 'schema', [])
        comparison = compare_schemas(destination_schema, schema)
        logging.debug('Schema comparison: %r', comparison)
        if comparison == SchemaComparisonResult.Incompatible:
            raise TableMergeError(
                'Can not merge tables {} and {} because of incompatible types'.format(
                    table, dst_table,
                )
            )

        #  Если целевая схема  не меньше, чем у исходный таблицы, то достаточно простого merge
        if comparison in (SchemaComparisonResult.Equal, SchemaComparisonResult.LeftGreater):
            ytw.run_merge(
                source_table=table,
                destination_table='<append=true>' + dst_table,
                mode='unordered',
                spec={
                    'schema_inference_mode': 'from_output' if destination_schema else 'auto',
                },
            )
        #  Если схемы несравнимы или схема исходной таблицы содержит поля, которых нет в целевой
        #  таблице, то нужна вспмоготальная таблица
        else:
            if schema:
                destination_schema = merge_schemas(destination_schema, schema)

            temp_table = yt_helpers.create_safe_temp_table(ytw, __file__, attributes={
                'schema': destination_schema,
                'optimize_for': 'scan',
            })

            ytw.run_merge(
                source_table=[table, dst_table],
                destination_table=temp_table,
            )

            ytw.remove(dst_table)
            ytw.move(temp_table, dst_table)

        processed.append(table)
        ytw.set_attribute(dst_table, 'processed', processed)


def _convert_records_to_dict(records):
    return {
        record['name']: record['type']
        for record in records
    }


def merge_schemas(left_schema, right_schema):
    left_schema = _convert_records_to_dict(left_schema)
    right_schema = _convert_records_to_dict(right_schema)

    for column, _type in iteritems(right_schema):
        if column in left_schema:
            if left_schema[column] != _type:
                raise ValueError('Cannot merge types {} and {} for column {}'.format(
                    left_schema[column], _type, column,
                ))
        else:
            left_schema[column] = _type

    return [
        {'name': name, 'type': _type}
        for name, _type in iteritems(left_schema)
    ]


def compare_schemas(left_schema, right_schema):
    left_schema = _convert_records_to_dict(left_schema)
    right_schema = _convert_records_to_dict(right_schema)
    used_from_left = set()
    right_is_greater = False
    left_is_greater = False
    for column, right_type in iteritems(right_schema):
        if column not in left_schema:
            right_is_greater = True
            continue

        if right_type != left_schema[column]:
            return SchemaComparisonResult.Incompatible

        used_from_left.add(column)

    left_is_greater = len(left_schema) > len(used_from_left)

    if right_is_greater and left_is_greater:
        return SchemaComparisonResult.BothGreater

    if right_is_greater:
        return SchemaComparisonResult.RightGreater

    if left_is_greater:
        return SchemaComparisonResult.LeftGreater

    return SchemaComparisonResult.Equal


def _create_table(ytw, name, image):
    logging.info('Creating table %s', name)
    attributes = {}
    schema = ytw.get_attribute(image, 'schema', [])
    if schema:
        attributes = {'schema': schema, 'optimize_for': 'scan'}
    ytw.create('table', name, attributes=attributes)

    for schema_name in ['_read_schema']:
        schema = ytw.get_attribute(image, schema_name, None)
        if schema is not None:
            ytw.set_attribute(name, schema_name, schema)
