# coding=utf-8
from __future__ import unicode_literals

"""
Декоратор sandbox задачи для выполнения действия только при обновлении sandbox ресурса.
Будет работать для задач запущенных из-под планеровщика.

Как применять.

1. В задаче добавить параметр, отвечающий за запуск механизма слежения за ресурсами,
   и в планировщике поставить его значение в true.
   По умолчанию параметр имеет имя act_only_on_resource_update.

2. В задаче добавить параметр, отвечающий за обновлением какого sandbox ресурса следить.
   По умолчанию параметр имеет имя watch_for_resource_type

3. Определить имя переменной контекста, в которую будет сохранена обработанная версия (id) ресурса

4. Добавить декоратор с соответствующими параметрами на сендбокс задачу
    @act_only_on_resource_update(
        enable_by_parameter='act_only_on_resource_update',
        resource_type_parameter='watch_for_resource_type',
        context_storage_name='_last_processed_resource_version',
    )
    class MyTask(...):
        class Parameters(sdk2.Task.Parameters):
            ...
            with sdk2.parameters.Group('Resource update detector'):
                act_only_on_resource_update = sdk2.parameters.Bool('Act only on resource update', default=False)
                watch_for_resource_type = sdk2.parameters.String('Watch for resource type',
                                                                 default='TRAVEL_DICT_RASP_REGION_PROD')
        ...

5. Настроить планеровщик
"""

import logging

from sandbox import sdk2
from sandbox.common.types import resource, task

logger = logging.getLogger(__name__)


def act_only_on_resource_update(
    enable_by_parameter='act_only_on_resource_update',
    resource_type_parameter='watch_for_resource_type',
    context_storage_name='_last_processed_resource_version',
):
    def decorator(obj):
        return act_only_on_resource_update_deco(obj, enable_by_parameter, resource_type_parameter, context_storage_name)
    return decorator


def act_only_on_resource_update_deco(obj, enable_by_parameter, resource_type_parameter, context_storage_name):
    original_on_execute = getattr(obj, 'on_execute')

    def on_execute(self, *args, **kwargs):
        enabled = getattr(self.Parameters, enable_by_parameter, False)
        if not enabled:
            logger.info('Act only on resource update is not enabled')
            return original_on_execute(self, *args, **kwargs)

        watch_for_resource_type = getattr(self.Parameters, resource_type_parameter)
        logger.info('Watch for resource %s', watch_for_resource_type)

        logger.info('Get current resource version')
        current_resource_version = get_current_resource_version(watch_for_resource_type)
        logger.info('Current resource version %s', current_resource_version)

        logger.info('Get last processed resource version')
        last_processed_resource_version = get_last_processed_resource_version(self.scheduler, context_storage_name)
        logger.info('Last processed resource version %s', last_processed_resource_version)

        if last_processed_resource_version and last_processed_resource_version == current_resource_version:
            logger.info('No need to act')
            result = None
        else:
            logger.info('Run action')
            result = original_on_execute(self, *args, **kwargs)

        logger.info('Save processed resource version')
        setattr(self.Context, context_storage_name, current_resource_version)
        logger.info('Done')

        return result

    setattr(obj, 'on_execute', on_execute)

    return obj


def get_current_resource_version(resource_type):
    last_resource = sdk2.Resource.find(type=resource_type, state=resource.State.READY,
                                       order='-id', limit=1).first()
    return last_resource.id if last_resource else None


def get_last_processed_resource_version(scheduler_id, context_storage_name):
    rasp_data_version = None
    if scheduler_id and scheduler_id > 0:
        logger.debug('Scheduler %s', scheduler_id)
        last_successful_task = sdk2.Task.find(scheduler=scheduler_id, status=task.Status.SUCCESS,
                                              order='-id', limit=1).first()
        if last_successful_task:
            logger.debug('Last successful task %s', last_successful_task)
            rasp_data_version = getattr(last_successful_task.Context, context_storage_name, None)
        else:
            logger.debug('No successful task')
    else:
        logger.debug('No scheduler')
    return rasp_data_version
