# -*- coding: utf-8 -*-
import logging
import os

import boto3
import fnmatch

logger = logging.getLogger(__name__)


class S3Client(object):
    def __init__(self, endpoint_url, log=logger):
        self._log = log
        self.endpoint_url = endpoint_url

        self._s3 = boto3.resource('s3', endpoint_url=endpoint_url)

    def bucket_exists(self, buck):
        buckets = [bucket for bucket in self._s3.buckets.all()]
        return buck in buckets

    def sync(self, sync_list, bucket_name):
        bucket = self._s3.Bucket(bucket_name)

        try:
            if not self.bucket_exists(bucket):
                self._log.info('Bucket does not exist, creating new bucket %s', bucket_name)
                self._s3.create_bucket(
                    Bucket=bucket_name,
                    CreateBucketConfiguration={
                        'LocationConstraint': 'us-east-1',
                    },
                )
            else:
                self._log.debug('Bucket %s was found in storage', bucket)
        except:
            self._log.exception(
                'Connection failed.\n'
                '\tPlease check for correct S3 credentials in environment variables '
                '(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY) for host %s',
                self.endpoint_url,
            )
            return

        for share, path, exclude in sync_list:
            try:
                self._log.info('cp %s => [%s]/%s', path, bucket, share)
                self.cp(bucket, share, path, excludes=exclude)
            except:
                self._log.exception('Could not copy %s to [%s]/%s', path, bucket, share)

    def cp(self, bucket, s3_key_prefix, path, excludes=None):
        dirname = os.path.dirname(path)
        basename = os.path.basename(path)
        self.copy_recursive(bucket, s3_key_prefix, dirname, basename, excludes)

    def copy_recursive(self, bucket, s3_key_prefix, path_prefix, path, excludes=None):
        realpath = os.path.join(path_prefix, path)
        if excludes is None:
            excludes = []
        for exc in excludes:
            if fnmatch.fnmatch(realpath, exc):
                self._log.debug('File %s is excluded', realpath)
                return

        if not os.path.exists(realpath):
            self._log.error('File %s not found', realpath)
            return

        if os.path.isdir(realpath):
            self._log.debug('File %s is a directory', realpath)
            for f in os.listdir(realpath):
                self.copy_recursive(bucket, s3_key_prefix, path_prefix, os.path.join(path, f))
            return

        self.copy_file(bucket, s3_key_prefix, path_prefix, path)

    def copy_file(self, bucket, s3_key_prefix, path_prefix, path):
        import magic

        realpath = os.path.join(path_prefix, path)
        s3_key = os.path.join(s3_key_prefix, path)
        self._log.debug('cp %s -> [%s]/%s', realpath, bucket, s3_key)
        try:
            mime = magic.from_file(realpath, mime=True)
            bucket.Object(s3_key).upload_file(
                realpath,
                ExtraArgs={
                    'ContentType': mime,
                }
            )
        except magic.MagicException:
            self._log.error("Could not detect MIME type for file %s", realpath)
            bucket.Object(s3_key).upload_file(realpath)
