# coding=utf-8
from __future__ import unicode_literals

import logging
import os.path
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.avia.base import AviaBaseTask

log = logging.getLogger(__name__)


class AviaLogUnknownFareCodes(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        SPECIFIC_DAY_KEY = 'specific_day'
        YESTERDAY_KEY = 'yesterday'

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Yt parameters'):
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token_vault = sdk2.parameters.String('YT token vault name', default='YT_TOKEN', required=True)

        ticket_daemon_api_host = sdk2.parameters.String(
            'ticket-daemon-api host',
            required=True,
            default='http://ticket-daemon-api.production.avia.yandex.net',
        )
        variants_log_directory = sdk2.parameters.String(
            'variants_log_directory',
            required=True,
            default='//home/logfeller/logs/avia-variants-log/1d',
        )
        with sdk2.parameters.RadioGroup('Log type') as log_type:
            log_type.values[YESTERDAY_KEY] = log_type.Value('Log for yesterday', default=True)
            log_type.values[SPECIFIC_DAY_KEY] = log_type.Value('Log for a specific day')

            with log_type.value[SPECIFIC_DAY_KEY]:
                date = sdk2.parameters.StrictString(
                    'Date YYYY-MM-DD',
                    regexp="\d{4}-\d{2}-\d{2}",
                    required=True,
                )

        output_table_path = sdk2.parameters.String(
            'Output table path',
            default='//home/avia/logs/avia-unknown-fare-codes-log',
        )
        vaults_owner = sdk2.parameters.String('Vaults owner', required=True, default_value='AVIA')

    def on_execute(self):
        super(AviaLogUnknownFareCodes, self).on_execute()
        from travel.avia.log_unknown_fare_codes.lib.pipeline import run_pipeline

        yt_token = sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yt_token_vault)

        if self.Parameters.log_type == self.Parameters.YESTERDAY_KEY:
            target_date = (datetime.now() - timedelta(days=1)).date()
        else:
            target_date = datetime.strptime(str(self.Parameters.date), '%Y-%m-%d').date()

        output_table = os.path.join(self.Parameters.output_table_path, target_date.strftime('%Y-%m-%d'))
        run_pipeline(
            self.Parameters.ticket_daemon_api_host,
            yt_token,
            self.Parameters.yt_proxy,
            self.Parameters.variants_log_directory,
            target_date,
            output_table,
        )
