# coding=utf-8
import logging
import os
from datetime import datetime

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.lib.merge_tables import merge_table, TableMergeError
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory

from sandbox.projects.avia.base import AviaBaseTask


DESTINATION_PREFIX_FORMAT = '//home/{}/logs/'
LOGS_PATH_PREFIX = '//home/logfeller/logs/'


def merge_logs_with_schema(ytw, src_path, dst_path, date_format,
                           min_date=None, max_date=None):
    if not ytw.exists(dst_path):
        ytw.create('map_node', dst_path, recursive=True)

    if not ytw.exists(src_path):
        logging.info('Source path [%s] does not exist, abort', src_path)
        return

    for table in ytw.search(src_path, node_type='table'):
        table_name = table.split('/')[-1]

        table_date = datetime.strptime(table_name, date_format)
        if min_date is not None and table_name < min_date:
            continue

        if max_date is not None and table_name > max_date:
            continue

        dst_table = os.path.join(dst_path, table_date.strftime('%Y-%m-%d'))
        try:
            merge_table(ytw, table, dst_table)
            logging.info('Done %s', table)
        except TableMergeError as e:
            logging.exception(e)
            continue


class AviaMergeLogsWithSchema(AviaBaseTask):
    """ Merge logs with schema """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('raven'),
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('YT Settings') as yt_settings:
            vaults_owner = sdk2.parameters.String('Token vault owner', required=True)
            vault_name = sdk2.parameters.String('Token vault name', required=True, default='YT_TOKEN')

        with sdk2.parameters.Group('Task settings') as task_settings:
            log_name = sdk2.parameters.String('Log name', required=True)
            project = sdk2.parameters.String('Project (e. g. avia, rasp)', required=True)

            with sdk2.parameters.RadioGroup('Mode') as mode:
                mode.values['30min'] = mode.Value('30 min', default=True)
                mode.values['d'] = mode.Value('1 day')
                mode.values['ultrafast'] = mode.Value('5 min')

            min_date = sdk2.parameters.String('Minimal date', required=False, default=None)
            max_date = sdk2.parameters.String('Maximal date', required=False, default=None)

    def on_execute(self):
        import yt.wrapper as yt
        logging.info('Start')

        self._configure_sentry()

        try:
            log_path = yt.ypath_join(LOGS_PATH_PREFIX, self.Parameters.log_name)
            mode = self.Parameters.mode
            logging.info('Mode: %r', mode)
            if mode == 'd':
                log_path = yt.ypath_join(log_path, '1d')
                date_format = '%Y-%m-%d'
            elif mode == '30min':
                log_path = yt.ypath_join(log_path, '30min')
                date_format = '%Y-%m-%dT%H:%M:%S'
            elif mode == 'ultrafast':
                log_path = yt.ypath_join(log_path, 'stream', '5min')
                date_format = '%Y-%m-%dT%H:%M:%S'
            else:
                logging.error('Unknown mode: %s', mode)
                raise Exception

            merge_logs_with_schema(
                YtClientFactory.create('hahn', sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.vault_name)),
                src_path=log_path,
                dst_path=yt.ypath_join(DESTINATION_PREFIX_FORMAT.format(self.Parameters.project), self.Parameters.log_name),
                date_format=date_format,
                min_date=self.Parameters.min_date or None,
                max_date=self.Parameters.max_date or None,
            )

        except Exception:
            logging.exception('ERROR')
            raise

    def _configure_sentry(self):
        configure_logging(get_sentry_dsn(self))
