# coding=utf-8
from __future__ import unicode_literals

import logging
import os
from collections import namedtuple
from datetime import datetime, timedelta, date
from functools import partial

import requests

from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib import yql_helpers as yqlh
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


logger = logging.getLogger(__name__)

CityInfo = namedtuple('CityInfo', ('title', 'geo_id', 'iata', 'sirena', 'title_accusative', 'title_locative'))

# Список городов, интересных ЯДМ:
INTERESTED_CITIES = {
    'c67': CityInfo('Томск', 67, [u'TOF'], ['ТСК'], 'Томска', 'Томске'),
    'c43': CityInfo('Казань', 43, [u'KZN'], ['КЗН'], 'Казани', 'Казани'),
    'c11': CityInfo('Рязань', 11, None, ['РЗН'], 'Рязани', 'Рязани'),
    'c51': CityInfo('Самара', 51, [u'KUF'], ['СМШ'], 'Самары', 'Самаре'),
    'c63': CityInfo('Иркутск', 63, [u'IKT'], ['ИКТ'], 'Иркутска', 'Иркутске'),
    'c76': CityInfo('Хабаровск', 76, [u'KHV'], ['ХБР'], 'Хабаровска', 'Хабаровске'),
    'c193': CityInfo('Воронеж', 193, [u'VOZ'], ['ВРН'], 'Воронежа', 'Воронеже'),
    'c237': CityInfo('Новокузнецк', 237, [u'NOZ'], ['НВК'], 'Новокузнецка', 'Новокузнецке'),
    'c75': CityInfo('Владивосток', 75, [u'VVO'], ['ВВО'], 'Владивостока', 'Владивостоке'),
    'c22': CityInfo('Калининград', 22, [u'KGD'], ['КЛД'], 'Калининграда', 'Калининграде'),
    'c213': CityInfo('Москва', 213, [u'MOW'], ['МОВ'], 'Москвы', 'Москве'),
    'c62': CityInfo('Красноярск', 62, [u'KJA'], ['КЯА'], 'Красноярска', 'Красноярске'),
    'c35': CityInfo('Краснодар', 35, [u'KRR'], ['КРР'], 'Краснодара', 'Краснодаре'),
    'c195': CityInfo('Ульяновск', 195, [u'ULY'], ['УЛК'], 'Ульяновска', 'Ульяновске'),
    'c172': CityInfo('Уфа', 172, [u'UFA'], ['УФА'], 'Уфы', 'Уфе'),
    'c197': CityInfo('Барнаул', 197, [u'BAX'], ['БАН'], 'Барнаула', 'Барнауле'),
    'c65': CityInfo('Новосибирск', 65, [u'OVB'], ['ОВБ'], 'Новосибирска', 'Новосибирске'),
    'c54': CityInfo('Екатеринбург', 54, [u'SVX'], ['ЕКБ'], 'Екатеринбурга', 'Екатеринбурге'),
    'c39': CityInfo('Ростов-на-Дону', 39, [u'ROV'], ['РОВ'], 'Ростова-на-Дону', 'Ростове-на-Дону'),
    'c50': CityInfo('Пермь', 50, [u'PEE'], ['ПРМ'], 'Перми', 'Перми'),
    'c23243': CityInfo('Нижний Новгород', 47, [u'GOJ'], ['НЖС'], 'Нижнего Новгорода', 'Нижнем Новгороде'),
    'c16': CityInfo('Ярославль', 16, [u'IAR'], ['ЯРЛ'], 'Ярославля', 'Ярославле'),
    'c45': CityInfo('Чебоксары', 45, [u'CSY'], ['ЧБЕ'], 'Чебоксар', 'Чебоксарах'),
    'c38': CityInfo('Волгоград', 38, [u'VOG'], ['ВГГ'], 'Волгограда', 'Волгограде'),
    'c194': CityInfo('Саратов', 194, [u'GSV'], ['СРО'], 'Саратова', 'Саратове'),
    'c64': CityInfo('Кемерово', 64, [u'KEJ'], ['КРВ'], 'Кемерово', 'Кемерово'),
    'c55': CityInfo('Тюмень', 55, [u'TJM'], ['ТЮМ'], 'Тюмени', 'Тюмени'),
    'c66': CityInfo('Омск', 66, [u'OMS'], ['ОМС'], 'Омска', 'Омске'),
    'c240': CityInfo('Тольятти', 240, [u'KUF'], ['СМШ'], 'Тольятти', 'Тольятти'),
    'c44': CityInfo('Ижевск', 44, [u'IJK'], ['ИЖВ'], 'Ижевска', 'Ижевске'),
    'c56': CityInfo('Челябинск', 56, [u'CEK'], ['ЧЛБ'], 'Челябинска', 'Челябинске'),
    'c2': CityInfo('Санкт-Петербург', 2, [u'LED'], ['СПТ'], 'Санкт-Петербурга', 'Санкт-Петербурге'),
}

MESSAGE_TEMPLATE = (
    'В России постепенно улучшается ситуация с {directions_type} авиасообщением. По данным Яндекс.Путешествий, за '
    '{comparison_period} количество выполненных рейсов из {{city_from_title_accusative}} {cities_type} увеличилось '
    'на {{percentage_diff}}%. Если на прошлой неделе их было {{before_flights_count}}, то на этой из города '
    '{{after_flights_count_phrase}}.'
)

LOCAL_FLIGHTS_WEEK_INFO_TEXT_TEMPLATE = MESSAGE_TEMPLATE.format(
    directions_type='внутренним', comparison_period='последнюю неделю', cities_type='в другие города'
)
LOCAL_FLIGHTS_MONTH_INFO_TEXT_TEMPLATE = MESSAGE_TEMPLATE.format(
    directions_type='внутренним', comparison_period='последний месяц', cities_type='в другие города'
)
FOREIGN_FLIGHTS_WEEK_INFO_TEXT_TEMPLATE = MESSAGE_TEMPLATE.format(
    directions_type='международным', comparison_period='последнюю неделю', cities_type='за рубеж'
)
FOREIGN_FLIGHTS_MONTH_INFO_TEXT_TEMPLATE = MESSAGE_TEMPLATE.format(
    directions_type='международным', comparison_period='последний месяц', cities_type='за рубеж'
)

LOCAL_FLIGHTS_WEEK_INFO_TITLE_TEMPLATE = 'Возобновление авиасообщения в {city_from_title_locative}'
LOCAL_FLIGHTS_MONTH_INFO_TITLE_TEMPLATE = 'Возобновление авиасообщения в {city_from_title_locative}'
FOREIGN_FLIGHTS_WEEK_INFO_TITLE_TEMPLATE = 'Возобновление авиасообщения в {city_from_title_locative}'
FOREIGN_FLIGHTS_MONTH_INFO_TITLE_TEMPLATE = 'Возобновление авиасообщения в {city_from_title_locative}'

LOCAL_FLIGHTS_WEEK_INFO_DESCRIPTION_TEMPLATE = \
    'Возобновление авиасообщения внтури страны(сравнение за неделю) в {city_from_title_locative} за {news_date}'
LOCAL_FLIGHTS_MONTH_INFO_DESCRIPTION_TEMPLATE = \
    'Возобновление авиасообщения внтури страны(сравнение за месяц) в {city_from_title_locative} за {news_date}'
FOREIGN_FLIGHTS_WEEK_INFO_DESCRIPTION_TEMPLATE = \
    'Возобновление международного авиасообщения(сравнение за неделю) в {city_from_title_locative} за {news_date}'
FOREIGN_FLIGHTS_MONTH_INFO_DESCRIPTION_TEMPLATE = \
    'Возобновление международного авиасообщения(сравнение за месяц) в {city_from_title_locative} за {news_date}'


def gen_after_flights_count_phrase(flights_count):
    variants = (
        'вылетел уже {} самолёт',
        'вылетело уже {} самолёта',
        'вылетело уже {} самолётов',
    )
    decimal_part = flights_count % 100
    if decimal_part // 10 != 1:
        if decimal_part % 10 == 1:
            return variants[0].format(flights_count)
        elif decimal_part % 10 in {2, 3, 4}:
            return variants[1].format(flights_count)
    return variants[2].format(flights_count)


class YangClient(object):
    def __init__(self, url, token, logger_):
        self._url = url
        self._token = token
        self._logger = logger_

    @property
    def _headers(self):
        return {
            'accept': 'application/json',
            'Authorization': 'OAuth ' + self._token,
        }

    def _check_response(self, response):
        if not response.ok:
            self._logger.info('Bad response %r. Content %s', response, response.content)
            response.raise_for_status()

    def create_pool(self, data):
        response = requests.post(
            self._url + '/api/v1/pools',
            headers=self._headers,
            json=data
        )
        self._check_response(response)
        return response.json()

    def get_pools(self, **params):
        response = requests.get(
            self._url + '/api/v1/pools',
            headers=self._headers,
            params=params
        )
        self._check_response(response)
        return response.json()

    def create_task_suites(self, data):
        response = requests.post(
            self._url + '/api/v1/task-suites?allow_defaults=true&open_pool=true',
            headers=self._headers,
            json=data
        )
        self._check_response(response)
        return response.json()

    def assignments(self, **params):
        response = requests.get(
            self._url + '/api/v1/assignments',
            headers=self._headers,
            params=params,
        )
        self._check_response(response)
        return response.json()


class NewsFeedTaskBase(object):
    _yql_client = None
    _yang_client = None
    _yt_client = None

    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('YQL settings') as yql_settings:
            yql_token = sdk2.parameters.YavSecret(
                'Yav-secret с YQL-токеном робота',
                default='sec-01e4q88q2jcz84c8dtwrm29z3r',
                required=True,
            )

        with sdk2.parameters.Group('YT settings') as yt_settings:
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret(
                'Yav-secret с YT-токеном робота',
                default='sec-01dfxmszhq27tk66hm107d0ycd',
                required=True,
            )

        with sdk2.parameters.Group('YANG settings') as yang_settings:
            yang_url = sdk2.parameters.String(
                'Yang/Toloka API url',
                required=True,
                default='https://yang.yandex-team.ru',
            )
            yang_oauth_token = sdk2.parameters.YavSecret(
                'Yav-secret с OAUTH-токеном робота для yang',
                default='sec-01egtaqxs9jn48hpzpjx8rqfgy',
                required=True,
            )
            yang_project_id = sdk2.parameters.String(
                'Yang news-feed project ID',
                required=True,
                default='9389',
            )

    def _get_yql_client(self):
        from yql.api.v1.client import YqlClient

        if self._yql_client is None:
            token = self.Parameters.yql_token.data()['token']
            self._yql_client = YqlClient(token=token)

        return self._yql_client

    def _get_yang_client(self):
        if self._yang_client is None:
            token = self.Parameters.yang_oauth_token.data()['token']
            self._yang_client = YangClient(self.Parameters.yang_url, token, logger)

        return self._yang_client

    def _get_yt_client(self):
        if self._yt_client is None:
            self._yt_client = YtClientFactory.create(
                self.Parameters.yt_proxy, self.Parameters.yt_token.data()['token']
            )

        return self._yt_client


class AviaNewsFeedRoutesDynamicPreparer(binary_task.LastBinaryTaskRelease, AviaBaseTask, NewsFeedTaskBase):
    """
        Регулярное вычисление количества рейсов для подготовки данных новостных лент.
        Отправка подготовленных текстов в yang.yandex-team.ru
    """
    class Parameters(NewsFeedTaskBase.Parameters):
        threshold = sdk2.parameters.Integer(
            'Порог по проценту увеличения числа рейсов для формирования новости',
            required=True,
            default=10
        )
        shared_flights_url = sdk2.parameters.String(
            'URL API единой базы авиарейсов',
            required=True,
            default='http://shared-flights.testing.avia.yandex.net'
        )

    def on_execute(self):
        super(AviaNewsFeedRoutesDynamicPreparer, self).on_execute()
        tasks = self._gen_tasks()
        if not tasks:
            logger.info('No tasks for publish')
            return
        pool_id = self._create_pool()
        self._publish_tasks(pool_id, tasks)

    def _read_query(self, query_name):
        from library.python import resource
        path = os.path.join(os.path.dirname(__file__), query_name)
        return resource.find(path)

    def _gen_tasks(self):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        yql_client = self._get_yql_client()
        yt_client = self._get_yt_client()
        temp_table = yt_client.create_temp_table()
        yql_query = self._read_query('query.sql')
        query = yql_client.query(
            yql_query,
            syntax_version=1,
            title='[YQL] Routes prepare count dynamic for news feed'
        )
        query_result = query.run(parameters=ValueBuilder.build_json_map({
            '$actual_flights_output_table': ValueBuilder.make_string(temp_table),
        }))
        logging.info('YQL Operation: %s', yqlh.get_yql_operation_url(query_result))
        query_result.wait_progress()

        if not query_result.is_success:
            yqlh.log_errors(query_result, logging)
            raise SandboxTaskFailureError('No results from YQL')

        self._fill_arrival_airport_codes(yt_client, temp_table)
        completed_temp_table = yt_client.create_temp_table()
        self._fill_country_id(yt_client, temp_table, completed_temp_table)

        yql_query = self._read_query('group_results_query.sql')
        query = yql_client.query(
            yql_query,
            syntax_version=1,
            title='[YQL] Routes count dynamic for news feed'
        )
        query_result = query.run(parameters=ValueBuilder.build_json_map({
            '$actual_flights_input_table': ValueBuilder.make_string(completed_temp_table),
        }))
        logging.info('YQL Operation: %s', yqlh.get_yql_operation_url(query_result))
        query_result.wait_progress()

        if not query_result.is_success:
            yqlh.log_errors(query_result, logging)
            raise SandboxTaskFailureError('No results from YQL')
        else:
            for result in query_result.get_results():
                if result.fetch_full_data():
                    logging.info('YQL query done')
                    result_news = []
                    for row in result.rows:
                        from_city_id, this_week_local_flights, last_week_local_flights, last_month_local_flights, \
                        this_week_foreign_flights, last_week_foreign_flights, last_month_foreign_flights = row

                        if from_city_id in INTERESTED_CITIES:
                            from_city_info = INTERESTED_CITIES[from_city_id]
                            logger.info('Generate news for %s. Data: %r', from_city_id, row)
                            one_week_internal_flights = self.gen_news(
                                from_city_info, last_week_local_flights, this_week_local_flights,
                                LOCAL_FLIGHTS_WEEK_INFO_TEXT_TEMPLATE, LOCAL_FLIGHTS_WEEK_INFO_TITLE_TEMPLATE,
                                LOCAL_FLIGHTS_WEEK_INFO_DESCRIPTION_TEMPLATE
                            )
                            # one_month_internal_flights = self.gen_news(
                            #     from_city_info, last_month_local_flights, this_week_local_flights,
                            #     LOCAL_FLIGHTS_MONTH_INFO_TEXT_TEMPLATE, LOCAL_FLIGHTS_MONTH_INFO_TITLE_TEMPLATE,
                            #     LOCAL_FLIGHTS_MONTH_INFO_DESCRIPTION_TEMPLATE
                            # )
                            one_week_foreign_flights = self.gen_news(
                                from_city_info, last_week_foreign_flights, this_week_foreign_flights,
                                FOREIGN_FLIGHTS_WEEK_INFO_TEXT_TEMPLATE, FOREIGN_FLIGHTS_WEEK_INFO_TITLE_TEMPLATE,
                                FOREIGN_FLIGHTS_WEEK_INFO_DESCRIPTION_TEMPLATE
                            )
                            # one_month_foreign_flights = self.gen_news(
                            #     from_city_info, last_month_foreign_flights, this_week_foreign_flights,
                            #     FOREIGN_FLIGHTS_MONTH_INFO_TEXT_TEMPLATE, FOREIGN_FLIGHTS_MONTH_INFO_TITLE_TEMPLATE,
                            #     FOREIGN_FLIGHTS_MONTH_INFO_DESCRIPTION_TEMPLATE
                            # )

                            # Todo: месячные новости пока не будем передавать.
                            generated_news = filter(None, [
                                one_week_internal_flights,
                                # one_month_internal_flights,
                                one_week_foreign_flights,
                                # one_month_foreign_flights
                            ])
                            logger.info('Generated news: %r', generated_news)
                            result_news.extend(generated_news)
                    logging.info('Generated %d news', len(result_news))
                    return result_news

    def _fill_arrival_airport_codes(self, yt_client, table_path):
        import yt.wrapper as yt

        updated_rows = []
        rows_for_update = {}

        def update_rows():
            if not rows_for_update:
                return

            response = requests.post(
                self.Parameters.shared_flights_url + '/api/v1/flights/',
                json={'flight': ','.join(rows_for_update)}
            )

            if not response.ok:
                logger.error('Bad response %r. Content %s', response, response.content)
                response.raise_for_status()

            for flight in response.json():
                flight_key = flight['departureDay'].replace('-', '') + flight['airlineCode'] + flight['number']
                row_for_update = rows_for_update.get(flight_key)
                if row_for_update:
                    row_for_update['arrival_airport'] = flight['airportToCode']
                    updated_rows.append(row_for_update)

            rows_for_update.clear()

        for row in yt_client.read_table(
            table_path,
            format=yt.JsonFormat(),
            raw=False
        ):
            if (
                row['arrival_airport'] is None and row['iata'] and row['flight_number']
                and row['flight_number'].isdigit() and row['flight_date']
            ):
                flight_key = row['flight_date'].replace('-', '') + row['iata'] + row['flight_number']
                rows_for_update[flight_key] = row

            if len(rows_for_update) >= 100:
                update_rows()
        update_rows()

        if updated_rows:
            logger.info('Add %d rows', len(updated_rows))
            yt_client.write_table(
                "<append=true>" + table_path,
                updated_rows,
            )

    def _get_search_codes_cache(self):
        import yt.wrapper as yt
        yt_client = self._get_yt_client()

        settlement_reference_table_path = '//home/rasp/reference/settlement'
        station_reference_table_path = '//home/rasp/reference/station'

        search_codes_cache = {}
        for row in yt_client.read_table(
            yt_client.TablePath(station_reference_table_path, columns=['sirena', 'iata', 'city_id', 'country_id']),
            format=yt.JsonFormat(),
            raw=False
        ):
            if row['sirena']:
                search_codes_cache[row['sirena']] = (row['city_id'], row['country_id'])
            if row['iata']:
                search_codes_cache[row['iata']] = (row['city_id'], row['country_id'])

        for row in yt_client.read_table(
            yt_client.TablePath(settlement_reference_table_path, columns=['sirena', 'iata', 'id', 'country_id']),
            format=yt.JsonFormat(),
            raw=False
        ):
            if row['sirena']:
                search_codes_cache[row['sirena']] = (row['id'], row['country_id'])
            if row['iata']:
                search_codes_cache[row['iata']] = (row['id'], row['country_id'])

        return search_codes_cache

    def _fill_country_id(self, yt_client, source_table, destination_table):
        def fill_reference(search_codes, record):
            if record['departure_airport'] and record['arrival_airport']:
                departure_airport = record['departure_airport'].decode('utf-8')
                arrival_airport = record['arrival_airport'].decode('utf-8')
                if departure_airport in search_codes and arrival_airport in search_codes:
                    record['from_city_id'], record['from_country_id'] = search_codes[departure_airport]
                    record['to_city_id'], record['to_country_id'] = search_codes[arrival_airport]
                    yield record

        yt_client.run_map(
            partial(fill_reference, self._get_search_codes_cache()),
            source_table=source_table,
            destination_table=destination_table,
        )

    def gen_news(self, city_info, before_count, after_count, text_template, title_template, description_template):
        if before_count == 0:
            if after_count == 0:
                percentage_diff = 0
            else:
                percentage_diff = 100
        else:
            percentage_diff = int((float(after_count) / before_count - 1) * 100)

        if percentage_diff >= self.Parameters.threshold:
            return {
                'geoId': str(city_info.geo_id),
                'title': title_template.format(
                    city_from_title_locative=city_info.title_locative,
                ),
                'text': text_template.format(
                    percentage_diff=percentage_diff,
                    before_flights_count=before_count,
                    after_flights_count_phrase=gen_after_flights_count_phrase(after_count),
                    city_from_title_accusative=city_info.title_accusative,
                ),
                'description': description_template.format(
                    city_from_title_locative=city_info.title_locative,
                    news_date=date.today(),
                )
            }

        return None

    def _create_pool(self):
        data = {
            "project_id": self.Parameters.yang_project_id,
            "private_name": "Авиабилеты. Новостная лента. %s" % datetime.now().date(),
            "may_contain_adult_content": False,
            "will_expire": (datetime.now() + timedelta(days=1)).strftime("%Y-%m-%dT%H:%M:%S"),
            "auto_close_after_complete_delay_seconds": 0,
            "reward_per_assignment": 0.01,
            "assignment_max_duration_seconds": 600,
            "auto_accept_solutions": True,
            "auto_accept_period_day": 21,
            "assignments_issuing_config": {
                "issue_task_suites_in_creation_order": False
            },
            "filter": {
                "and": [
                    {
                        "or": [
                            {
                                "category": "skill",
                                "key": "30444",
                                "operator": "GT",
                                "value": "0"
                            }
                        ]
                    }
                ]
            },
            "defaults": {
                "default_overlap_for_new_task_suites": 1
            },
            "priority": 0,
            "owner": {
                "id": "bd349925f49d1bf9490d76754a92ee71",
                "myself": True,
                "company_id": "1"
            },
            "type": "REGULAR",
            "status": "OPEN",
        }
        result = self._get_yang_client().create_pool(data)

        return result['id']

    def _publish_tasks(self, pool_id, tasks):
        data = [{'pool_id': pool_id, 'tasks': [{'input_values': task}]} for task in tasks]
        self._get_yang_client().create_task_suites(data)


class AviaNewsFeedRoutesDynamicPublisher(binary_task.LastBinaryTaskRelease, AviaBaseTask, NewsFeedTaskBase):
    """Перенос подтвержденных новостей из yang.yandex-team.ru в Yt-таблицу"""
    class Parameters(NewsFeedTaskBase.Parameters):
        with sdk2.parameters.Group('YT tables settings') as yt_tables_settings:
            approved_news_table_path = sdk2.parameters.String(
                'YT table path for approved news',
                default='//home/avia/news-feed/approved',
                required=True,
            )
            rejected_news_table_path = sdk2.parameters.String(
                'YT table path for rejected news',
                default='//home/avia/news-feed/rejected',
                required=True,
            )

        news_creation_max_shift = sdk2.parameters.Integer(
            'Maximum number of days since the news creation',
            default=3,
            required=True,
        )

    def on_execute(self):
        super(AviaNewsFeedRoutesDynamicPublisher, self).on_execute()

        pools = self._get_pools()
        for pool in pools:
            approved_news, rejected_news = self._get_news(pool)
            self.update_yt_tables(approved_news, self.Parameters.approved_news_table_path)
            self.update_yt_tables(rejected_news, self.Parameters.rejected_news_table_path)

    def _get_pools(self):
        yang_client = self._get_yang_client()
        left_created_date = datetime.now() - timedelta(days=self.Parameters.news_creation_max_shift)
        data = yang_client.get_pools(
            project_id=self.Parameters.yang_project_id,
            created_gt=str(left_created_date.date()),
        )

        if len(data.get('items', [])) == 0:
            logger.warning('Empty items. Response %r', data)
            return []

        pools = []
        for pool in data['items']:
            pools.append(pool)

        return pools

    def _get_news(self, pool):
        pool_id = pool['id']
        yang_client = self._get_yang_client()
        data = yang_client.assignments(pool_id=pool_id)

        if len(data.get('items', [])) == 0:
            logger.warning('Empty items. Response %r', data)
            return (), ()

        approved_news = []
        bad_news = []
        for tasks_page in data['items']:
            if tasks_page['status'] != 'ACCEPTED':
                logger.warning('Tasks is not accepted: %r', tasks_page)
                continue

            for task, solution in zip(tasks_page['tasks'], tasks_page['solutions']):
                news = {
                    'id': task['id'],
                    'title': task['input_values']['title'],
                    'text': task['input_values']['text'],
                    'geoIds': str(task['input_values']['geoId']),
                    'createdAt': pool.get('created', ''),
                    'acceptedAt': tasks_page.get('accepted', ''),
                    'userId': tasks_page['user_id'],
                }
                if solution['output_values']['quality'] == 'BAD':
                    logger.info('Bad news. Info: %r', news)
                    bad_news.append(news)
                elif solution['output_values']['quality'] == 'EDIT':
                    news.update({
                        'title': solution['output_values']['title'],
                        'text': solution['output_values']['text'],
                        'geoIds': str(solution['output_values']['geoId']),
                    })
                    logger.info('Edited news. Info: %r', news)
                    approved_news.append(news)
                elif solution['output_values']['quality'] == 'OK':
                    logger.info('OK news. Info: %r', news)
                    approved_news.append(news)
                else:
                    logger.warning('Unknown quality %s', solution['output_values']['quality'])

        return approved_news, bad_news

    def update_yt_tables(self, news, table_path):
        import yt.wrapper as yt
        yt_client = self._get_yt_client()

        if not yt_client.exists(table_path):
            schema = [
                {'name': 'id', 'type': 'string'},
                {'name': 'title', 'type': 'string'},
                {'name': 'text', 'type': 'string'},
                {'name': 'geoIds', 'type': 'string'},
                {'name': 'createdAt', 'type': 'string'},
                {'name': 'acceptedAt', 'type': 'string'},
                {'name': 'updatedAt', 'type': 'string'},
                {'name': 'userId', 'type': 'string'},
            ]

            yt_client.create(
                'table',
                table_path,
                attributes={'schema': schema, 'optimize_for': 'scan'},
                recursive=True
            )

        actual_news_ids = {
            row['id'] for row in yt_client.read_table(
                yt.TablePath(table_path, columns=('id',)),
                format=yt.JsonFormat(),
                raw=False
            )
        }
        updated_at = datetime.utcnow().isoformat()
        news_for_update = [
            dict(one_news, updatedAt=updated_at)
            for one_news in news
            if one_news['id'] not in actual_news_ids
        ]
        if news_for_update:
            logger.info('Add %d news', len(news_for_update))
            yt_client.write_table(
                "<append=true>" + table_path,
                news_for_update,
            )
