# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import difflib
import json
import logging
from datetime import datetime, timedelta
from itertools import chain
from os import path

import sandbox.sandboxsdk.environments as sdk_environments
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.common.types.notification import Transport

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib import logs

logger = logging.getLogger(__name__)


class AVIA_FARE_DIFF(sdk2.Resource):
    pass


class AviaNotifyAboutFareChanges(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    """
    Send email about partner fare changes
    """
    _yt_client = None

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Map reduce settings') as mr_block:
            yt_cluster = sdk2.parameters.String('MapReduce cluster', default='arnold', required=True)
            yt_token = sdk2.parameters.YavSecret("yt_token", required=True, default="sec-01dfxmszhq27tk66hm107d0ycd")
            reports_folder = sdk2.parameters.String('Watson reports folder', required=True,
                                                    default='//home/watson/parsers_data')
            avia_company_prefix = sdk2.parameters.String(
                'Common prefix for avia reports (relative to watson report folder)', required=True,
                default='annashpak')

        with sdk2.parameters.Group('Task settings') as task_block:
            notifications_recipients = sdk2.parameters.List(
                'Notifications recipients',
                required=True,
                default=[],
                value_type=sdk2.parameters.String)

            report_ttl = sdk2.parameters.Integer('Ignore reports older than N days:', required=True, default=7)

        ext_params = binary_task.binary_release_parameters(none=True)
        vaults_owner = sdk2.parameters.String('Sentry secret vault owner', required=True, default_value='AVIA')

    class Requirements(sdk2.Requirements):
        # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

        environments = [sdk_environments.PipEnvironment('yandex-yt', version='0.10.8'),
                        sdk_environments.PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
                        sdk_environments.PipEnvironment('raven')]

    def _get_yt_client(self):
        if self._yt_client is None:
            import yt.wrapper
            self._yt_client = yt.wrapper.YtClient(
                proxy=self.Parameters.yt_cluster,
                token=self.Parameters.yt_token.data()['token']
            )
            self._yt_json_format = yt.wrapper.JsonFormat()
        return self._yt_client

    def on_execute(self):
        super(AviaNotifyAboutFareChanges, self).on_execute()
        logs.configure_logging(logs.get_sentry_dsn(self))
        logger.info('Start')
        self._get_yt_client()
        fares = self.get_fares(self.Parameters.reports_folder, self.Parameters.avia_company_prefix)
        resource_url, partners = self.build_report_files(fares)
        if partners:
            self.send_resource_report(resource_url, partners)
        logger.info('Done')

    @staticmethod
    def get_filter(base_directory, prefix):
        def filter_(path_):
            return path_.startswith(path.join(base_directory, prefix))

        return filter_

    def get_fares(self, base_directory, prefix):
        fares = []
        for folder in self._yt_client.search(base_directory, node_type='map_node', depth_bound=1,
                                             path_filter=self.get_filter(base_directory, prefix)):
            logger.info('processing folder %s', folder)
            fare = self.get_fares_for_partner(folder)
            if fare:
                fares.append(fare)
        return fares

    def get_fares_for_partner(self, folder):
        batch_folder = path.join(folder, 'batch_mode')
        if not self._yt_client.exists(batch_folder):
            logger.info('%s does not have "batch_mode" folder. Skipping.', folder, )
            return
        reports = list(
            self._yt_client.search(batch_folder, node_type='table', depth_bound=1,
                                   path_filter=self.is_valid_report_table))
        if len(reports) < 2:
            logger.info('Not enough recent reports for partner %s', folder)
            return
        reports.sort(reverse=True)
        before = next(self._yt_client.read_table(reports[1], format='json'))['Result']
        after = next(self._yt_client.read_table(reports[0], format='json'))['Result']
        return {'partner': folder, 'before': before, 'after': after}

    def is_valid_report_table(self, table_path):
        pattern = '%Y-%m-%dT%H:%M:%S'
        try:
            date = table_path.split('/')[-1]
            report_date = datetime.strptime(date, pattern)
        except ValueError:
            return False

        return report_date >= (datetime.today() - timedelta(days=self.Parameters.report_ttl))

    def send_resource_report(self, resource_url, partners):
        text = chain(['Following partners have changed fares:'], partners, ['See diffs {}'.format(resource_url)])
        self.server.notification(
            subject="Fare change alert",
            body="\n".join(text),
            recipients=self.Parameters.notifications_recipients,
            transport=Transport.EMAIL,
            urgent=False
        )

    def build_report_files(self, partners):
        reports = []
        for partner in partners:
            diff = self.get_diff(partner['before'], partner['after'])
            if diff:
                reports.append({'partner': partner['partner'], 'diff': diff})

        if not reports:
            return None, None

        resource = AVIA_FARE_DIFF(
            self, "fare_diffs", "fare_diffs"
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        for report in reports:
            folder = report['partner'].split('/')[-1] + '.html'
            resource_data.path.joinpath(folder).write_bytes(report['diff'].encode('utf-8'))

        return resource.http_proxy, [report['partner'] for report in reports]

    @staticmethod
    def get_diff(before, after):
        differ = difflib.HtmlDiff(wrapcolumn=60)
        before = json.dumps(before, indent=4, sort_keys=True, ensure_ascii=False, encoding='utf8').split('\n')
        after = json.dumps(after, indent=4, sort_keys=True, ensure_ascii=False, encoding='utf8').split('\n')
        if before != after:
            return differ.make_file(before, after, context=True)
