# -*- coding: utf-8 -*-
import logging
from collections import Counter
from datetime import datetime, timedelta
from urlparse import urljoin

from sandbox.projects.avia.lib.yt_helpers import YtClientFactory

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
import sandbox.common.types.notification as ctn
import sandbox.common.types.misc as ctm

from sandbox.projects.avia.base import AviaBaseTask

log = logging.getLogger(__name__)


class AviaNotifyAboutUnknownCompanies(AviaBaseTask):
    class Requirements(sdk2.Task.Requirements):
        # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

        environments = (
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
        )
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):

        yt_proxy = sdk2.parameters.String('Reference YT proxy', required=True, default_value='hahn')
        vaults_owner = sdk2.parameters.String('Vaults owner', required=True, default_value='AVIA')
        unknown_companies_log_yt_path = sdk2.parameters.String(
            'Unknown companies log YT path',
            required=True,
            default_value='//home/logfeller/logs/avia-unknown-companies-log/1d/',
        )
        yt_token_vault_name = sdk2.parameters.String('YT-token vault name', required=True, default='YT_TOKEN')
        notifications_recipients = sdk2.parameters.List(
            'Notifications recipients',
            required=True,
            default=[],
            value_type=sdk2.parameters.String,
        )

    def on_execute(self):
        self._yt = YtClientFactory.create(
            self.Parameters.yt_proxy,
            sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yt_token_vault_name)
        )
        yesterday = datetime.today() - timedelta(days=1)
        self._day = yesterday.strftime('%Y-%m-%d')
        unknown_company_counters = self._get_unknown_company_counters()
        message_body = self._build_message_body(unknown_company_counters)
        self._send_notifications(message_body)

    def _get_unknown_company_counters(self):
        import yt.wrapper as yt_wrapper

        table_path = urljoin(self.Parameters.unknown_companies_log_yt_path, self._day)
        return Counter(
            record['company_iata']
            for record in self._yt.read_table(
                table_path,
                format=yt_wrapper.format.JsonFormat()
            )
        )

    def _build_message_body(self, unknown_company_counters):
        headers = '<tr><th>IATA</th><th>Вхождений в лог</th><tr>'
        return (
            '<p>Дневной лог: '
            '<a href="https://yt.yandex-team.ru/{proxy}/navigation?path={path}">'
            'https://yt.yandex-team.ru/{proxy}/navigation?path={path}'
            '</a></p>'.format(
                proxy=self.Parameters.yt_proxy,
                path=urljoin(self.Parameters.unknown_companies_log_yt_path, self._day),
            ) +
            '<table style="border: 1px solid black;border-collapse: collapse;">' +
            '<caption>Неизвестные авиакомпании за {}</caption>'.format(self._day) +
            headers +
            self.format_unknown_companies(unknown_company_counters)
            + '</table>'
        )

    @staticmethod
    def format_unknown_companies(unknown_company_counters):
        return ''.join(
            '<tr>'
            '<td style="border: 1px solid black;border-collapse: collapse;">{}</td>'
            '<td style="border: 1px solid black;border-collapse: collapse;">{}</td>'
            '</tr>'.format(company, counter)
            for company, counter in unknown_company_counters.most_common()
        )

    def _send_notifications(self, message_body):
        log.info('Sending message to recipients: %s', self.Parameters.notifications_recipients)
        log.info('Message body: %s', message_body)
        self.server.notification(
            subject='Неизвестные авиакомпании за {}'.format(self._day),
            body=message_body,
            recipients=self.Parameters.notifications_recipients,
            transport=ctn.Transport.EMAIL,
            type='html',
        )
