# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import tempfile
from datetime import date, timedelta

from sandbox import sdk2
from sandbox.projects.common import binary_task

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory
from sandbox.projects.avia.price_prediction.build_good_price.pg_helpers import update_pg_variant_price_stats
from sandbox.projects.avia.lib.pg_helpers import get_psycopg2_conn_string
from sandbox.projects.avia.price_prediction.build_good_price.yt_helpers import (
    export_yt_table_to_dsv, update_price_storage, update_shows, take_popular, update_quantiles
)

log = logging.getLogger(__name__)

DEFAULT_GOOD_SERVICES = [
    'avia-travel',
    'mavia-travel',
]

PG_TABLE_NAME = 'variant_price_stats'
EXPORT_TO_PG_COLUMNS = [
    'point_from_type',
    'point_from_id',
    'point_to_type',
    'point_to_id',
    'route_uid',
    'days_to_flight',
    'departure_weekday',
    'q33',
    'q67',
]


class AviaPricePredictionBuildGoodPrice(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 4096  # 4096 Megs
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group('Binary task release parameters') as binary_task_release_parameters:
            binary_release = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('YT settings') as yt_settings:
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret('Yav-secret с YT-токеном робота', required=True,
                                                 default='sec-01dfxmszhq27tk66hm107d0ycd')

        with sdk2.parameters.Group('Build good price parameters') as build_good_price_parameters:
            days = sdk2.parameters.Integer('Number of days to process', default=1, required=True)
            good_services = sdk2.parameters.List('Good services', required=True, default=DEFAULT_GOOD_SERVICES)
            output_yt_prefix = sdk2.parameters.String('Output YT prefix', required=True,
                                                      default='//home/avia/tmp/price-prediction/')

        with sdk2.parameters.Group('Postgresql settings') as pg_settings:
            pg_host = sdk2.parameters.String('Postgresql host name', required=True,
                                             default='c-mdb9sssbmtcje8gtvlrc.rw.db.yandex.net')
            pg_port = sdk2.parameters.String('Postgresql port', required=True, default=6432)
            pg_user = sdk2.parameters.String('Postgresql user', default='avia', required=True)
            pg_dbname = sdk2.parameters.String('Postgresql dbname', default='price-prediction', required=True)
            pg_password_vault = sdk2.parameters.YavSecret('Postgresql password', required=True,
                                                          default='sec-01e84cae2rjpe3pvwjrt9c0phb')

    def on_execute(self):
        import psycopg2
        from psycopg2.extras import LoggingConnection

        log.info('Start build good prices')

        yt_price_storage_table = self.Parameters.output_yt_prefix + 'price_storage_days'
        yt_popular_price_storage_table = self.Parameters.output_yt_prefix + 'popular_price_storage_days'
        yt_shows_table = self.Parameters.output_yt_prefix + 'shows'
        yt_quantiles_popular_table = self.Parameters.output_yt_prefix + 'price_quantiles_days_popular'

        ytc = YtClientFactory.create(
            proxy=self.Parameters.yt_proxy,
            token=self.Parameters.yt_token.data()['token']
        )

        pg_connection_string = get_psycopg2_conn_string(
            host=self.Parameters.pg_host,
            port=self.Parameters.pg_port,
            db_name=self.Parameters.pg_dbname,
            user=self.Parameters.pg_user,
            password=self.Parameters.pg_password_vault.data()['password'],
        )

        start_date = date.today() - timedelta(days=self.Parameters.days)
        end_date = date.today() - timedelta(days=1)  # Yesterday
        log.info('Update shows')
        update_shows(ytc, yt_shows_table, start_date, end_date)

        log.info('Updating price storage: %s', yt_price_storage_table)
        good_services = self.Parameters.good_services
        log.info('Good services %s', good_services)
        update_price_storage(ytc, start_date, end_date, good_services, yt_price_storage_table)

        log.info('Taking popular flights')
        take_popular(
            ytc,
            yt_shows_table,
            yt_price_storage_table,
            yt_popular_price_storage_table,
        )

        log.info('Calculating quantiles')
        update_quantiles(
            ytc,
            yt_popular_price_storage_table,
            yt_quantiles_popular_table
        )

        log.info('Export to pg')
        with tempfile.TemporaryDirectory() as temp_path:
            yt_data_filename = os.path.join(temp_path, 'yt_data')
            export_yt_table_to_dsv(ytc, yt_quantiles_popular_table, EXPORT_TO_PG_COLUMNS, yt_data_filename)

            with psycopg2.connect(pg_connection_string, connection_factory=LoggingConnection) as connection:
                connection.initialize(log)
                update_pg_variant_price_stats(connection, yt_data_filename, PG_TABLE_NAME, EXPORT_TO_PG_COLUMNS)
