# coding=utf-8
import logging
from datetime import datetime

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory

from sandbox.projects.avia.base import AviaBaseTask


class AviaRemoveOldYtTempTables(AviaBaseTask):
    """ Remove old YT temp tables """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('raven'),
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('YT Settings') as yt_settings:
            vaults_owner = sdk2.parameters.String('Token vault owner', default='robot-avia-api-pilot', required=True)
            vault_name = sdk2.parameters.String('Token vault name', required=True, default='YT_TOKEN')

        with sdk2.parameters.Group('Task settings') as task_settings:
            path = sdk2.parameters.String('YT path', required=True, default='//home/avia/tmp')
            days = sdk2.parameters.Integer('Days (delete older than)', required=True, default=1)

    def on_execute(self):
        logging.info('Start')

        configure_logging(get_sentry_dsn(self))

        yt_client = YtClientFactory.create(
            'hahn',
            sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.vault_name)
        )

        now = datetime.now()
        removed = 0

        for table in yt_client.search(self.Parameters.path, node_type='table'):
            try:
                modification_time = datetime.strptime(
                    yt_client.get_attribute(table, 'modification_time'),
                    '%Y-%m-%dT%H:%M:%S.%fZ',
                )
                if (now - modification_time).days >= self.Parameters.days:
                    logging.info('Removing %s', table)
                    yt_client.remove(table)
                    removed += 1
            except Exception as e:
                logging.warning('Error processing table %s: %s', table, e)

        logging.info('Removed %d', removed)
