import os
import logging
import subprocess

import sandbox.sdk2.helpers
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.avia.base import AviaBaseTask

CONFIGURE_DOCKER = (
    """#!/usr/bin/env bash

set -eux

export DEBIAN_FRONTEND=noninteractive

echo "# Path to docker root: $1"

ATTEMPTS=1
COMMAND="apt-get update"
until $COMMAND || [ $ATTEMPTS -eq 4 ]; do
    echo "Failed \"$COMMAND\", attempt=$(( ATTEMPTS++ )), sleep 5 seconds..."
    sleep 5
done

ATTEMPTS=1
COMMAND="apt-get install --yes --force-yes apt-transport-https ca-certificates curl software-properties-common"
until $COMMAND || [ $ATTEMPTS -eq 4 ]; do
    echo "Failed \"$COMMAND\", attempt=$(( ATTEMPTS++ )), sleep 5 seconds..."
    sleep 5
done

curl -fsSL https://download.docker.com/linux/ubuntu/gpg | apt-key add -
apt-key fingerprint 0EBFCD88
add-apt-repository "deb [arch=amd64] https://download.docker.com/linux/ubuntu $(lsb_release -cs) stable"

ATTEMPTS=1
COMMAND="apt-get update"
until $COMMAND || [ $ATTEMPTS -eq 4 ]; do
    echo "Failed \"$COMMAND\", attempt=$(( ATTEMPTS++ )), sleep 5 seconds..."
    sleep 5
done

ATTEMPTS=1
COMMAND="apt-get install -y --force-yes docker-ce=17.09.1* unbound-config-local64"
until $COMMAND || [ $ATTEMPTS -eq 4 ]; do
    echo "Failed \"$COMMAND\", attempt=$(( ATTEMPTS++ )), sleep 5 seconds..."
    sleep 5
done

default_gw_if=$(ip -6 r s | grep -m 1 ^default | awk '{print $5}')

ip -6 r s
ip a

mkdir -p "$1"
service docker stop
ip link del docker0 || true
rm -rf /var/lib/docker

echo > /etc/default/docker
echo 'DOCKER_OPTS="-D -s overlay -H unix://'"$1"'/docker.sock --ip-forward=false --ipv6 --dns 2a02:6b8:0:3400::1023 """
    """--dns 2a02:6b8:0:3400::5005 --fixed-cidr-v6 fd00::/8 --graph '"$1"'"' | tee -a /etc/default/docker

service docker start

ip -6 r s
ip a

ip6tables -t nat -A POSTROUTING -s fd00::/8 -j MASQUERADE
echo 2 > /proc/sys/net/ipv6/conf/${default_gw_if}/accept_ra
echo 1 > /proc/sys/net/ipv6/conf/all/forwarding

ip -6 r s
ip a

docker -H unix://$1/docker.sock info
docker -H unix://$1/docker.sock run -i --rm ubuntu:14.04 ifconfig eth0
"""
)

DEFAULT_LXC_CONTAINER_ID = 698163198
ENV_VARIABLES_FILENAME = 'environment_variables'


class AviaRunAdminScript(AviaBaseTask):
    """
    Task that runs specified admin script.
    """

    class Requirements(sdk2.Task.Requirements):
        privileged = True
        environments = (
            environments.PipEnvironment('requests'),
            environments.PipEnvironment('ujson'),
        )

    class Parameters(sdk2.Task.Parameters):

        _container = sdk2.parameters.Container(
            'Environment LXC container resource',
            default_value=DEFAULT_LXC_CONTAINER_ID,
            required=True
        )

        vault_owner = sdk2.parameters.String('Vault owner', required=True)
        specific_docker_image_url = sdk2.parameters.String('Specific docker image (url)', required=False)
        command = sdk2.parameters.String('Command relative to admin project root', required=True)

        with sdk2.parameters.Group('Qloud settings') as qloud_settings:
            qloud_environment = sdk2.parameters.String('Environment', required=True, default_value='production')
            qloud_domain = sdk2.parameters.String('Domain', required=True, default_value='qloud-ext.yandex-team.ru')
            qloud_oauth_token_vault_name = sdk2.parameters.String(
                'OAuth token vault name',
                required=True,
                default_value='AVIA_QLOUD_OAUTH_TOKEN',
            )

        with sdk2.parameters.Group('Docker settings') as docker_settings:
            docker_login = sdk2.parameters.String('Login', required=True, default_value='avia')
            docker_oauth_token_vault_name = sdk2.parameters.String(
                'OAuth token vault name',
                required=True,
                default_value='AVIA_DOCKER_REGISTRY_OAUTH_TOKEN',
            )

        with sdk2.parameters.Group('Environment') as envitonment_settings:
            environment_variables = sdk2.parameters.Dict('Environment variables')
            mysql_user = sdk2.parameters.String(
                'Avia MySQL user (AVIA_MYSQL_USER)', required=True, default_value='rasp',
            )
            mysql_password_vault_name = sdk2.parameters.String(
                'Avia MySQL password vault name', required=True, default_value='AVIA_MYSQL_PASSWORD',
            )
            yt_token_vault_name = sdk2.parameters.String(
                'Avia YT token vault name', required=True, default_value='YT_TOKEN',
            )
            yql_token_vault_name = sdk2.parameters.String(
                'Avia YQL token vault name', required=True, default_value='YQL_TOKEN',
            )

            secret_key_vault_name = sdk2.parameters.String(
                'Avia Admin Django SECRET_KEY vault name', required=True, default_value='AVIA_ADMIN_SECRET_KEY',
            )

            secret_salt_vault_name = sdk2.parameters.String(
                'Avia Admin Django SECRET_SALT vault name', required=True, default_value='AVIA_ADMIN_SECRET_SALT',
            )

            statface_token_vault_name = sdk2.parameters.String(
                'STATFACE_TOKEN vault name', required=False, default_value='',
            )

            sentry_dsn = sdk2.parameters.String('Sentry DSN (SENTRY_DSN)', required=True)

        with sdk2.parameters.Group('MDB settings') as mdb_block:
            mysql_cluster_id = sdk2.parameters.String(
                'Avia MySQL cluster id (AVIA_MYSQL_CLUSTER_ID)', required=False, default_value='',
            )
            mdb_token = sdk2.parameters.String(
                'Avia mdb token vault name', required=True, default_value='AVIA_MDB_API_TOKEN',
            )

        with sdk2.parameters.Group('MDS settings') as mds_block:
            mds_endpoint = sdk2.parameters.String('MDS endpoint url')
            mds_bucket = sdk2.parameters.String('MDS bucket name')
            aws_access_key_vault_name = sdk2.parameters.String('AWS access secret key vault name')
            aws_access_key_id_vault_name = sdk2.parameters.String('AWS key id vault name')

    @staticmethod
    def _get_current_admin_docker_image_url(qloud_domain, qloud_oauth_token, environment):
        from requests import Session
        from requests.adapters import HTTPAdapter
        from urllib3.util.retry import Retry
        import ujson

        session = Session()
        adapter = HTTPAdapter(
            max_retries=Retry(
                total=3,
                read=3,
                connect=3,
                backoff_factor=0.1,
                status_forcelist=(500, 502, 503, 504),
                method_whitelist=('GET',),
            ),
        )
        session.mount('https://', adapter)

        application = 'avia.avia-admin.{env}'.format(env=environment)
        request_url = 'https://{qloud_domain}/api/v1/environment/dump/{application}'.format(
            qloud_domain=qloud_domain,
            application=application,
        )
        response = session.get(
            request_url,
            headers={'Authorization': 'OAuth ' + qloud_oauth_token}
        )
        response.raise_for_status()

        environment_info = ujson.loads(response.text)
        return environment_info['components'][0]['properties']['repository']

    def _configure_docker(self):
        self.docker_root = os.path.join(str(self.path()), 'docker_root')
        self.docker_sock = 'unix://{}'.format(os.path.join(self.docker_root, 'docker.sock'))
        logging.info('Docker root %s', self.docker_root)
        docker_script_file = os.path.join(str(self.path()), 'docker_script')

        with open(docker_script_file, 'w') as fscript:
            fscript.write(CONFIGURE_DOCKER)

        with sandbox.sdk2.helpers.ProcessLog(self, logger='docker_prepare') as pl:
            retcode = subprocess.Popen(
                'bash {} {}'.format(docker_script_file, self.docker_root),
                shell=True,
                stderr=pl.stdout,
                stdout=pl.stdout,
            ).wait()
            if retcode == 0:
                if self._require_internet():
                    logging.info('Checking internet connection from docker')
                    subprocess.check_call(
                        'docker -H {} run -i --rm ubuntu:14.04 ping6 2001:4860:4860::8888 -c4'.format(
                            self.docker_sock
                        ),
                        shell=True,
                        stdout=pl.stdout,
                        stderr=pl.stderr,
                    )
                logging.info('Docker successfully configured')
            else:
                self.docker_root = None
                raise Exception('Docker configuration failed')

        os.environ['DOCKER_HOST'] = self.docker_sock
        os.environ['DOCKER_API_VERSION'] = self._get_docker_api_version()
        os.remove(docker_script_file)

    @staticmethod
    def _require_internet():
        return False

    def _get_docker_api_version(self):
        result = subprocess.check_output(
            ['docker', '-H', self.docker_sock, 'version', '--format', '{{.Server.APIVersion}}']
        )
        return str(result).strip()

    def _create_environment_variables(self):
        environment_variables = self.Parameters.environment_variables
        environment_variables['COMMAND'] = self.Parameters.command
        environment_variables['RUN_ONLY_ONE_COMMAND'] = 'True'
        environment_variables['QLOUD_DATACENTER'] = 'sas'
        environment_variables['SENTRY_DSN'] = self.Parameters.sentry_dsn
        environment_variables['AVIA_MYSQL_USER'] = self.Parameters.mysql_user
        environment_variables['AVIA_MYSQL_PASSWORD'] = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.mysql_password_vault_name
        )
        environment_variables['YT_TOKEN'] = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.yt_token_vault_name
        )
        environment_variables['YQL_TOKEN'] = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.yql_token_vault_name
        )

        environment_variables['AVIA_ADMIN_SECRET_KEY'] = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.secret_key_vault_name,
        )

        environment_variables['AVIA_ADMIN_SECRET_SALT'] = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.secret_salt_vault_name,
        )

        if self.Parameters.statface_token_vault_name:
            environment_variables['STATFACE_TOKEN'] = sdk2.Vault.data(
                self.Parameters.vault_owner,
                self.Parameters.statface_token_vault_name,
            )

        environment_variables['MDS_S3_ENDPOINT_URL'] = self.Parameters.mds_endpoint
        environment_variables['MDS_S3_AVIA_ADMIN_BUCKET'] = self.Parameters.mds_bucket
        if self.Parameters.aws_access_key_id_vault_name:
            environment_variables['AWS_ACCESS_KEY_ID'] = sdk2.Vault.data(
                self.Parameters.vault_owner,
                self.Parameters.aws_access_key_id_vault_name
            )
        if self.Parameters.aws_access_key_vault_name:
            environment_variables['AWS_SECRET_ACCESS_KEY'] = sdk2.Vault.data(
                self.Parameters.vault_owner,
                self.Parameters.aws_access_key_vault_name
            )

        environment_variables['AVIA_MYSQL_CLUSTER_ID'] = self.Parameters.mysql_cluster_id
        environment_variables['AVIA_MDB_API_TOKEN'] = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.mdb_token,
        )

        with open(ENV_VARIABLES_FILENAME, 'w') as file_output:
            for env_key, env_value in environment_variables.iteritems():
                file_output.write('{key}={value}\n'.format(key=env_key, value=env_value))

    def _login_docker(self, docker_oauth_token):
        run_process(
            'sudo docker -H {docker_sock} login -u {login} -p {token} registry.yandex.net'.format(
                docker_sock=self.docker_sock,
                login=self.Parameters.docker_login,
                token=docker_oauth_token,
            ),
            shell=True,
            log_prefix='Login to Docker'
        )

    def _run_docker(self, admin_docker_image_url):
        run_process(
            'sudo docker -H {docker_sock} run --env-file {env_file} {image_url}'.format(
                image_url=admin_docker_image_url,
                docker_sock=self.docker_sock,
                env_file=ENV_VARIABLES_FILENAME,
            ),
            shell=True,
            log_prefix='Running container'
        )

    def _clean_up(self):
        if self.docker_root is not None:
            run_process(
                'sudo docker -H {docker_sock} system prune -a -f'.format(docker_sock=self.docker_sock),
                shell=True,
                log_prefix='Cleaning up'
            )

    def _get_docker_image_url(self):
        if self.Parameters.specific_docker_image_url:
            return self.Parameters.specific_docker_image_url
        qloud_token = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.qloud_oauth_token_vault_name
        )
        return self._get_current_admin_docker_image_url(
            self.Parameters.qloud_domain,
            qloud_token,
            self.Parameters.qloud_environment
        )

    def on_execute(self):
        logging.info('Execution started')
        docker_image_url = self._get_docker_image_url()
        logging.info('Admin docker image url: {url}'.format(url=docker_image_url))

        docker_oauth_token = sdk2.Vault.data(
            self.Parameters.vault_owner,
            self.Parameters.docker_oauth_token_vault_name
        )
        self._configure_docker()

        logging.info('Docker image url: {url}'.format(url=docker_image_url))
        logging.info('Docker login: {login}'.format(login=self.Parameters.docker_login))
        logging.info('Docker oauth token: {token}'.format(token=str(docker_oauth_token)))

        self._create_environment_variables()
        self._login_docker(docker_oauth_token)
        self._run_docker(docker_image_url)
        self._clean_up()
