# coding=utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import sys

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.resource_update_detector import act_only_on_resource_update


@act_only_on_resource_update(
    enable_by_parameter='act_only_on_resource_update',
    resource_type_parameter='watch_for_resource_type',
    context_storage_name='_last_processed_resource_version',
)
class AviaRunAdminScriptBinary(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    """
    Task that runs specified admin script.
    """

    class Requirements(sdk2.Requirements):
        cores = 2
        disk_space = 30 * 1024  # 30 Gb
        ram = 1 * 1024  # 1 Gb

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Script settings') as script_settings:
            script_main = sdk2.parameters.String('Path to script main function', required=True,
                                                 description='travel.avia.admin.avia_scripts.sync_with_rasp.sync_all.main')
            script_params = sdk2.parameters.String('Script parameters', required=False, description='--exclude=pointsynonym -v')

        with sdk2.parameters.Group('Environment') as environment_settings:
            environment_variables = sdk2.parameters.Dict('Environment variables')

        with sdk2.parameters.Group('Admin settings') as admin_settings:
            admin_secret = sdk2.parameters.YavSecret('Avia Admin secret', required=True, default='sec-01eqzvnvf5bbjspektbpy9q3zc')
            admin_secret_key_key = sdk2.parameters.String('Avia Admin SECRET_KEY key', required=True, default='key')
            admin_secret_salt_key = sdk2.parameters.String('Avia Admin SECRET_SALT key', required=True, default='salt')
            yav_token_secret = sdk2.parameters.YavSecret('Yav token secret', default='sec-01dhbtw3k9zf21zst01csf4fkm')
            sentry_dsn = sdk2.parameters.String('Sentry DSN (SENTRY_DSN)', required=True)

        with sdk2.parameters.Group('MySQL settings') as mysql_settings:
            mysql_user = sdk2.parameters.String('Avia MySQL user (AVIA_MYSQL_USER)', required=True, default_value='avia')
            mysql_secret = sdk2.parameters.YavSecret(
                'Avia MySQL secret',
                required=True,
                description='testing: sec-01dvzx8m6rcvpb6dz2hcagpvvn, production: sec-01dvzx5stx4qtdjmd4jcv8dvzy',
            )

        with sdk2.parameters.Group('MDB settings') as mdb_block:
            mysql_cluster_id = sdk2.parameters.String(
                'Avia MySQL cluster id (AVIA_MYSQL_CLUSTER_ID)', required=False, default_value='',
            )
            mdb_token_secret = sdk2.parameters.YavSecret('Avia mdb token secret', required=True, default='sec-01ea9g76t94txbgpr0fhshy7y4')

        with sdk2.parameters.Group('YT settings') as yt_settings:
            yt_token_secret = sdk2.parameters.YavSecret('Avia YT token secret', required=True, default='sec-01dfxmszhq27tk66hm107d0ycd')
            yql_token_secret = sdk2.parameters.YavSecret('Avia YQL token secret', required=True, default='sec-01e4q88q2jcz84c8dtwrm29z3r')

        with sdk2.parameters.Group('Statface settings') as statface_settings:
            statface_token_secret = sdk2.parameters.YavSecret(
                'Avia statface token secret', required=False, default='', description='sec-01eqzv7dm35bj8gz1k9bkv8xwc'
            )

        with sdk2.parameters.Group('MDS settings') as mds_block:
            mds_endpoint = sdk2.parameters.String('MDS endpoint url')
            mds_bucket = sdk2.parameters.String('MDS bucket name')

            aws_secret = sdk2.parameters.YavSecret(
                'MDS S3 AWS secret', description='testing: sec-01dsey00pc7fs9mdjxgt8nn3xy, production: sec-01dsexxyfkmr6spccdc1q385wp'
            )
            aws_secret_key_key = sdk2.parameters.String('AWS secret AWS_SECRET_ACCESS_KEY key', required=False, default='aws_access_key_id')
            aws_secret_id_key = sdk2.parameters.String('AWS secret AWS_ACCESS_KEY_ID key', required=False, default='aws_secret_access_key')

        with sdk2.parameters.Group('Resource update detector') as resource_update_detector_block:
            act_only_on_resource_update = sdk2.parameters.Bool('Act only on resource update', default=False)
            watch_for_resource_type = sdk2.parameters.String('Watch for resource type', default='TRAVEL_DICT_RASP_REGION_PROD')

    def fill_task_tracker_config(self):
        """
        Define different source and host in solomon metric for different scripts
        """
        super(AviaRunAdminScriptBinary, self).fill_task_tracker_config()
        script_name = self.Parameters.script_main

        common_prefix = 'travel.avia.admin.'
        if script_name.startswith(common_prefix):
            script_name = script_name[len(common_prefix):]

        common_suffix = '.main'
        if script_name.endswith(common_suffix):
            script_name = script_name[:-len(common_suffix)]
        elif script_name.endswith('.manage') and self.Parameters.script_params:
            actual_script_name = self.Parameters.script_params.split()[0]
            script_name = '{}.{}'.format(script_name, actual_script_name)

        script_name = script_name.replace('_', '')

        self.task_tracker_config.source = '{}.{}'.format(self.task_tracker_config.source, script_name)
        self.task_tracker_config.host = '{}.{}'.format(self.task_tracker_config.host, script_name)
        logging.info('Set solomon source %s', self.task_tracker_config.source)

    def _create_environment_variables(self):
        environment_variables = self.Parameters.environment_variables
        environment_variables['QLOUD_DATACENTER'] = 'sas'
        environment_variables['SENTRY_DSN'] = self.Parameters.sentry_dsn
        environment_variables['AVIA_MYSQL_USER'] = self.Parameters.mysql_user
        environment_variables['AVIA_MYSQL_PASSWORD'] = self.get_secret_value(self.Parameters.mysql_secret, 'password')
        environment_variables['YT_TOKEN'] = self.get_secret_value(self.Parameters.yt_token_secret, 'token')
        environment_variables['YQL_TOKEN'] = self.get_secret_value(self.Parameters.yql_token_secret, 'token')
        environment_variables['AVIA_ADMIN_SECRET_KEY'] = self.get_secret_value(self.Parameters.admin_secret, self.Parameters.admin_secret_key_key)
        environment_variables['AVIA_ADMIN_SECRET_SALT'] = self.get_secret_value(self.Parameters.admin_secret, self.Parameters.admin_secret_salt_key)
        environment_variables['AVIA_ADMIN_YAV_OAUTH_TOKEN'] = self.get_secret_value(self.Parameters.yav_token_secret, 'token')

        if self.Parameters.statface_token_secret:
            environment_variables['STATFACE_TOKEN'] = self.get_secret_value(self.Parameters.statface_token_secret, 'token')

        environment_variables['MDS_S3_ENDPOINT_URL'] = self.Parameters.mds_endpoint
        environment_variables['MDS_S3_AVIA_ADMIN_BUCKET'] = self.Parameters.mds_bucket

        if self.Parameters.aws_secret:
            environment_variables['AWS_ACCESS_KEY_ID'] = self.get_secret_value(self.Parameters.aws_secret, self.Parameters.aws_secret_id_key)
            environment_variables['AWS_SECRET_ACCESS_KEY'] = self.get_secret_value(self.Parameters.aws_secret, self.Parameters.aws_secret_key_key)

        environment_variables['AVIA_MYSQL_CLUSTER_ID'] = self.Parameters.mysql_cluster_id
        environment_variables['AVIA_MDB_API_TOKEN'] = self.get_secret_value(self.Parameters.mdb_token_secret, 'oauth')

        return environment_variables

    def on_execute(self):
        import six
        from django.utils.module_loading import import_string

        logging.info('Execution started')

        logging.info('Creating env variables')
        environment_variables = self._create_environment_variables()

        logging.info('Set env variables')
        for variable, value in six.iteritems(environment_variables):
            os.environ[variable] = value

        logging.info('Run script')
        old_argv = sys.argv
        params = self.Parameters.script_params.split() if self.Parameters.script_params else []
        sys.argv = [self.Parameters.script_main] + params
        try:
            main = import_string(self.Parameters.script_main)
            main()
        finally:
            sys.argv = old_argv

        logging.info('Script run: done!')
