import tarfile

from sandbox.projects.common.kosher_release import find_release
from sandbox import sdk2
from sandbox.projects.yabs.infra.run_binary_resource import RunBinaryResource


class AviaRunReleasedBinaryResource(RunBinaryResource):
    class Parameters(RunBinaryResource.Parameters):
        with sdk2.parameters.Group('Resource:'):
            resource_by_id = sdk2.parameters.Bool('resource_by_id', default=False)
            unpack = sdk2.parameters.Bool('unpack resource', default=False)
            with resource_by_id.value[False]:
                resource_type = sdk2.parameters.String('resource_type', required=True)
                resource_attrs = sdk2.parameters.Dict('resource_attrs')
                resource_stage = sdk2.parameters.String("resource_stage", required=True)
            with resource_by_id.value[True]:
                resource = sdk2.parameters.Resource('resource', required=True)

    def on_execute(self):
        env = {}
        for k, v in self.Parameters.env_vars.items():
            if v.startswith('sec-'):
                ver, var = v.split(':', 1)
                v = sdk2.yav.Secret(ver).data()[var]
            elif v.startswith('vault:'):
                _, owner, name = v.split(':', 2)
                v = sdk2.Vault.data(owner, name)
            env[k] = v

        if self.Parameters.resource_by_id:
            binary = self.Parameters.resource
        else:
            resource_id = find_release(self.Parameters.resource_type, self.Parameters.resource_stage)
            if resource_id is None:
                raise RuntimeError("No resource found")
            binary = sdk2.Resource.find(id=resource_id).first()
            self.set_info(
                'There are <a href="https://sandbox.yandex-team.ru/resource/{}">resource</a>'.format(binary.id),
                do_escape=False,
            )
        binary_path = sdk2.ResourceData(binary).path.as_posix()
        if self.Parameters.unpack:
            work_dir = str(self.path())
            tar_path = str(sdk2.ResourceData(binary).path)
            with tarfile.open(tar_path) as tar:
                tar.extractall(path=work_dir)
            binary_path = str(self.path())

        try:
            ansi_out = sdk2.helpers.subprocess.check_output(
                self.Parameters.cmd.format(resource=binary_path),
                shell=True,
                env=env,
                stderr=sdk2.helpers.subprocess.STDOUT,
            )
            self.attach_pretty_log(ansi_out)
        except sdk2.helpers.subprocess.CalledProcessError as error:
            ansi_out = error.output
            self.attach_pretty_log(ansi_out)
            message = '{} failed, with status code {}'.format(error.cmd, error.returncode)
            raise RuntimeError(message)
