# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib import yql_helpers as yqlh
from sandbox.projects.avia.lib.yt_helpers import copy_table_to_weak, YtClientFactory


YESTERDAY = datetime.today() - timedelta(days=1)


WIZARD_SHOW_AVIA_METRICS_QUERY = '''
PRAGMA yson.Strict = '0';

DECLARE $Date AS Date;
DECLARE $OutputPath AS String;

$FormatDate = DateTime::Format('%Y-%m-%d');
$DateStr = $FormatDate($Date);

$wizardQueryPath = 'logs/avia-wizard-query-log/1d/' || $DateStr;
$wizardPointParsePath = 'logs/avia-wizard-point-parse-log/1d/' || $DateStr;
$wizardResultPath = 'logs/avia-wizard-response-log/1d/' || $DateStr;
$wizardShowPath = 'logs/avia-wizard-show-log/1d/' || $DateStr;


$BoolToString = ($x) -> {
    RETURN IF($x, '1', '0');
};


$GetPrimaryYakMetrics = ($yak) -> {
    RETURN AsList('wiz_avia_primary_yak=' || $BoolToString($yak));
};

$GetSecondaryYakMetrics = ($yak, $afterPreloader) -> {
    $name = 'wiz_avia_secondary_yak_' || IF($afterPreloader, 'after_preloader', 'before_preloader');
    RETURN AsList($name || '=' || $BoolToString($yak));
};

$GetTimingMetrics = ($elapsed) -> {
    -- Bucket bordres (0, 20, 140, 2000, inf) in milliseconds
    -- elapsed in seconds
    $wiz_avia_source_timing_20 = $BoolToString($elapsed <= 20.0 / 1000);
    $wiz_avia_source_timing_140 = $BoolToString($elapsed <= 140.0 / 1000);
    $wiz_avia_source_timing_2000 = $BoolToString($elapsed <= 2000.0 / 1000);
    $wiz_avia_source_timing_slow = $BoolToString($elapsed > 2000.0 / 1000);

    RETURN AsList(
        'wiz_avia_source_timing_20=' || $wiz_avia_source_timing_20,
        'wiz_avia_source_timing_140=' || $wiz_avia_source_timing_140,
        'wiz_avia_source_timing_2000=' || $wiz_avia_source_timing_2000,
        'wiz_avia_source_timing_slow=' || $wiz_avia_source_timing_slow
    )
};


$GetMetrics = ($wasOk, $afterPreloader, $yak, $isDynamic, $elapsed) -> {
    RETURN ListExtend(
        IF(
            $wasOk,
            IF(
                $isDynamic,
                $GetSecondaryYakMetrics($yak, $afterPreloader),
                $GetPrimaryYakMetrics($yak)
            ),
            ListCreate(DataType("String"))
        ),
        AsList('wiz_avia_ok=' || $BoolToString($wasOk)),
        IF(
            $elapsed IS NULL,
            ListCreate(DataType("String")),
            $GetTimingMetrics($elapsed)
        )
    )
};

$GetMetricLine = ($wasOk, $afterPreloader, $yak, $isDynamic, $elapsed) -> {
    RETURN String::JoinFromList(
        $GetMetrics($wasOk, $afterPreloader, $yak, $isDynamic, $elapsed),
        '\t'
    )
};

$get_user_id = ($serp_uuid, $yandexuid, $icookie) -> {
    RETURN 'y' || if($icookie is not null and $icookie!='' and len($icookie) < 30, $icookie, $yandexuid);
};

$is_cpa = ($partner) -> {
    RETURN $partner in ('aeroflot', 'pobeda') or ('dohop' <= $partner  and $partner < 'dohop_z');
};
$is_airline = ($partner) -> {
    RETURN $partner in ('s_seven', 'utair', 'csa', 'rusline', 'azimuth');
};
$is_agent = ($partner) -> {
    RETURN not $is_cpa($partner) and not $is_airline($partner);
};

$tariff_price_value = ($tariff) -> {
    return IF(
        Yson::Contains($tariff, 'value'),
        Yson::LookupDouble($tariff, 'value'),
        Yson::LookupDouble($tariff, 'price')
    )
};

$Shows = (
    SELECT
        job_id,
        1 as showed,
        count(*) as cnt,
        sum(transfer_count) as transfer_count,
        sum($tariff_price_value(tariff)) as price,
        min($tariff_price_value(tariff)) as min_price,
        max($tariff_price_value(tariff)) as max_price,
        count_if(from_aviacompany) as from_aviacompany,
        count_if(index < 5) as cnt5,
        sum_if(transfer_count, index < 5) as transfer_count5,
        sum_if($tariff_price_value(tariff), index < 5) as price5,
        count_if(from_aviacompany and index < 5) as from_aviacompany5,
        count_if(partner == 'aeroflot') as cnt_aeroflot,
        count_if($is_cpa(partner)) as cnt_cpa,
        count_if($is_airline(partner)) as cnt_airline,
        count_if($is_agent(partner)) as cnt_agent
    FROM $wizardShowPath
    GROUP BY job_id
);

$make_parameter = ($metrica, $value) -> {
    RETURN if($value is null, '', '\twiz_avia_source_offers_'||$metrica||'='||cast($value as String));
};

INSERT INTO
    $OutputPath
WITH TRUNCATE

SELECT
    $get_user_id(WeakField(query.serp_uuid, 'string'), query.yandexuid, query.icookie) AS key,
    ' stat_queries' AS subkey,
    query.reqid || '\t' || $GetMetricLine(
        r.status = 'OK', -- wasOk
        WeakField(query.after_preloader, 'bool') ?? false,  -- after_preloader
        s.job_id IS NULL, -- yak
        query.avia_dynamic IS NOT NULL, -- isDynamic
        r.elapsed -- elapsed
    )
    || $make_parameter('page', query.page)
    || $make_parameter('showed', s.showed)
    || $make_parameter('count', s.cnt)
    || $make_parameter('transfer_count', s.transfer_count)
    || $make_parameter('price', s.price)
    || $make_parameter('min_price', s.min_price)
    || $make_parameter('max_price', s.max_price)
    || $make_parameter('from_aviacompany', s.from_aviacompany)
    || $make_parameter('count5', s.cnt5)
    || $make_parameter('transfer_count5', s.transfer_count5)
    || $make_parameter('price5', s.price5)
    || $make_parameter('from_aviacompany5', s.from_aviacompany5)
    || $make_parameter('count_aeroflot', s.cnt_aeroflot)
    || $make_parameter('count_cpa', s.cnt_cpa)
    || $make_parameter('count_airlines', s.cnt_airline)
    || $make_parameter('count_agent', s.cnt_agent)
    AS value
FROM $wizardQueryPath AS query
LEFT JOIN $wizardPointParsePath AS pp
ON query.job_id = pp.job_id
LEFT JOIN $wizardResultPath AS r
ON query.job_id = r.job_id
LEFT JOIN $Shows AS s
ON query.job_id = s.job_id
WHERE
        pp.from_point_key IS NOT NULL
    AND pp.from_point_key != ""
    AND pp.to_point_key IS NOT NULL
    AND pp.to_point_key != ""
    AND r.type != 'flight'
    AND query.dev != 1
    AND query.reqid != "SUB_REQID"
    AND $get_user_id(WeakField(query.serp_uuid, 'string'), query.yandexuid, query.icookie) IS NOT NULL
ORDER BY key, subkey
'''


class SendAviaShowMetricsToABT(AviaBaseTask):
    """
    Send Avia wizards show metrics to ABT.
    See https://st.yandex-team.ru/RASPTICKETS-14464 for metric definition
    """

    _yt_client = None
    _yql_client = None

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('requests'),
            PipEnvironment('yql')
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Map reduce settings') as mr_block:
            mr_cluster = sdk2.parameters.String('MapReduce cluster', default='hahn', required=True)
            vaults_owner = sdk2.parameters.String('MapReduce user', required=True)
            mr_dir = sdk2.parameters.String('Directory', required=True, default='//home/avia/abt_wizard')
            yt_token_vault_name = sdk2.parameters.String('YT Token vault name', required=True, default='YT_TOKEN')

        with sdk2.parameters.Group('YQL settings') as yql_block:
            yql_token_vault_name = sdk2.parameters.String('YQL token vault name', required=True, default='YQL_TOKEN')

        with sdk2.parameters.Group('Date settings') as date_block:
            left_date = sdk2.parameters.String('Start date (default yesterday)', required=False)
            right_date = sdk2.parameters.String('End date (default yesterday)', required=False)

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            debug_run = sdk2.parameters.Bool('Debug run', default=False, required=True)

    @property
    def yt_client(self):
        if self._yt_client is None:
            self._yt_client = YtClientFactory.create(
                proxy=self.Parameters.mr_cluster,
                token=sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yt_token_vault_name),
            )

        return self._yt_client

    @property
    def yql_client(self):
        if self._yql_client is None:
            from yql.api.v1.client import YqlClient
            self._yql_client = YqlClient(
                token=sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yql_token_vault_name),
                db=self.Parameters.mr_cluster,
            )

        return self._yql_client

    def create_tag(self, date, cluster='hahn'):
        return 'cluster={cluster}_calc_avia_wizard_show_metrics_{date}'.format(
            cluster=cluster,
            date=date.strftime('%Y%m%d'),
        )

    def push_tag(self, tag):
        import requests
        return requests.get('http://rem-bernstein.n.yandex-team.ru/set_tag?tag=' + tag)

    def create_and_push_tag(self, date):
        tag = self.create_tag(date, self.Parameters.mr_cluster)
        logging.info('Tag: %s', tag)
        if not self.Parameters.debug_run:
            self.push_tag(tag)

    def process_date(self, date, output_path):
        logging.info('Process date: %s', date.strftime('%Y-%m-%d'))

        temp_table = self.create_temp_table_with_abt_metrics(date)
        output_table = '{}/{}'.format(output_path, date.strftime('%Y%m%d'))
        copy_table_to_weak(self.yt_client, from_table=temp_table, to_table=output_table, logger=logging)
        self.create_and_push_tag(date)

    def create_temp_table_with_abt_metrics(self, date):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        temp_table = self.yt_client.create_temp_table()
        r = self.yql_client.query(
            WIZARD_SHOW_AVIA_METRICS_QUERY,
            syntax_version=1,
            title='[YQL] Generating Avia wizard show metrics',
        ).run(parameters=ValueBuilder.build_json_map({
            '$OutputPath': ValueBuilder.make_string(temp_table),
            '$Date': ValueBuilder.make_date(date),
        }))

        logging.info('YQL Operation: %s', r.share_url)
        r.wait_progress()

        if not r.is_success:
            yqlh.log_errors(r, logging)
            raise SandboxTaskFailureError('YQL query failed')

        logging.info('YQL query done in %s table', temp_table)
        return temp_table

    def on_execute(self):
        logging.info('Start')

        left_date = self._parse_date(self.Parameters.left_date).date()
        logging.info('Left date: %s', left_date.strftime('%Y-%m-%d'))
        right_date = self._parse_date(self.Parameters.right_date).date()
        logging.info('Right date: %s', right_date.strftime('%Y-%m-%d'))

        if not self.yt_client.exists(self.Parameters.mr_dir):
            self.yt_client.create('map_node', self.Parameters.mr_dir, recursive=True)

        current_date = left_date
        while current_date <= right_date:
            self.process_date(current_date, self.Parameters.mr_dir)
            current_date += timedelta(days=1)

        logging.info('End')

    def _parse_date(self, s):
        if not s:
            return YESTERDAY
        try:
            return datetime.today() + timedelta(days=int(s))
        except ValueError:
            return datetime.strptime(s, '%Y-%m-%d')
