# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.common import binary_task, solomon

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.datetime_helpers import get_utc_now

logger = logging.getLogger(__name__)

DEFAULT_SKIP_TAGS = [
    'DEPRECATED',
]


class AviaSendSchedulerStatsToMonitoring(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    """
    Send schedulers last run time to solomon
    """
    SANDBOX_API_LIMIT = 100
    SANDBOX_API_URL = 'https://sandbox.yandex-team.ru/api/v1.0'

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Binary task release parameters') as binary_task_release_parameters:
            binary_release = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Solomon settings') as solomon_settings:
            solomon_project = sdk2.parameters.String('Solomon project', required=True, default='avia')
            solomon_cluster = sdk2.parameters.String('Solomon cluster', required=True, default='sandbox_testing')
            solomon_service = sdk2.parameters.String('Solomon service', required=True, default='last_run_time')
            solomon_token_secret = sdk2.parameters.YavSecret(
                "Yav secret id of robot oauth token for solomon",
                default="sec-01dwy9p37p1e8j94c5kp9zqmw1"
            )

        with sdk2.parameters.Group('Sandbox settings') as sandbox_settings:
            scheduler_owner = sdk2.parameters.String('scheduler owner', required=True, default='AVIA')
            scheduler_tags = sdk2.parameters.List(
                'Tags filter',
                required=False,
                default=[],
                value_type=sdk2.parameters.String,
            )
            skip_tags = sdk2.parameters.List(
                'Skip tags. Do not process scheduler if it has one of this tags',
                required=False,
                default=DEFAULT_SKIP_TAGS,
                value_type=sdk2.parameters.String,
            )

    class Requirements(sdk2.Requirements):
        # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # exactly 1 core
        ram = 128  # 512M or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    def _send_data_to_solomon(self, data, token):
        params = {
            'project': self.Parameters.solomon_project,
            'cluster': self.Parameters.solomon_cluster,
            'service': self.Parameters.solomon_service,
        }
        logger.info('Shard: {}'.format(params))
        for i in range(0, len(data), 1000):
            chunk = data[i:i+1000]
            logger.info(chunk)
            solomon.push_to_solomon_v2(token, params, chunk)

    def _get_stats(self, sandbox_group, tags):
        import pytz
        import requests

        params = {
            'owner': sandbox_group,
            'tags': ','.join(tags),
            'all_tags': True,
            'limit': self.SANDBOX_API_LIMIT,
            'offset': 0,
        }

        total = None
        stats = []
        while total is None or params['offset'] < total:
            scheduler_endpoint = '{}/scheduler'.format(self.SANDBOX_API_URL)
            logger.info('querying {} with params {}'.format(scheduler_endpoint, params))
            response = requests.get(scheduler_endpoint, params=params, timeout=30)
            response.raise_for_status()
            data = response.json()
            now = get_utc_now()
            for scheduler in data['items']:
                if not scheduler['time']['last']:
                    continue
                if self._scheduler_has_skip_tag(scheduler):
                    continue

                # time format is inconsistent in api, so crop microseconds, we don't need them anyway
                last_run_string = scheduler['time']['last'][:19]
                last_run = datetime.strptime(last_run_string, '%Y-%m-%dT%H:%M:%S')
                last_run_utc = pytz.utc.localize(last_run)
                seconds_since_last_run = (now - last_run_utc).total_seconds()
                stat = {
                    'value': seconds_since_last_run,
                    'ts': now.strftime('%Y-%m-%dT%H:%M:%SZ'),
                    'labels': {
                        'sensor': 'seconds_since_last_run',
                        'scheduler_id': scheduler['id'],
                        'task_type': scheduler['task']['type'],
                    }
                }
                stats.append(stat)
                if scheduler['schedule']['repetition']:
                    repetition_interval = scheduler['schedule']['repetition']['interval']
                    if not repetition_interval and scheduler['schedule']['repetition']['weekly']:
                        # some schedulers do not have repetition interval, but instead run only on some days of the week
                        # to keep things simple, do not check which days they should run,
                        # just check that they run at least once per week
                        repetition_interval = 60 * 60 * 24 * (8 - len(scheduler['schedule']['repetition']['weekly']))

                    if repetition_interval:
                        stat = {
                            'value': repetition_interval,
                            'ts': now.strftime('%Y-%m-%dT%H:%M:%SZ'),
                            'labels': {
                                'sensor': 'repetition_interval',
                                'scheduler_id': scheduler['id'],
                                'task_type': scheduler['task']['type'],
                            }
                        }
                        stats.append(stat)

            params['offset'] += params['limit']
            if not total:
                total = data['total']

        return stats

    def on_execute(self):
        token = self.Parameters.solomon_token_secret.data()['oauth']
        logging.info('Start')
        stats = self._get_stats(self.Parameters.scheduler_owner, self.Parameters.scheduler_tags)
        if stats:
            logger.info('Sending to Solomon')
            self._send_data_to_solomon(stats, token)
        else:
            logger.info('No data to send')

        logger.info('Done')

    def _scheduler_has_skip_tag(self, scheduler):
        if not self.Parameters.skip_tags:
            return False

        tags = set(scheduler.get('task', {}).get('tags', []))
        return bool(set(self.Parameters.skip_tags) & tags)
