# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import timedelta, datetime

import sandbox.sandboxsdk.environments as sdk_environments
from pytz import UTC
from sandbox import sdk2
from sandbox.projects.common import solomon
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib import logs, yql_helpers as yqlh

logger = logging.getLogger(__name__)

YESTERDAY = datetime.today() - timedelta(days=1)

QUERY_AVIA = """
USE hahn;

DECLARE $date_minus AS String;
DECLARE $date AS String;
DECLARE $abt_log_dir AS String;
DECLARE $abt_user_session_dir AS String;
DECLARE $output_dir AS String;

$abt_log = $abt_log_dir||'/'||$date;
$abt_user_session = $abt_user_session_dir||'/'||$date||'/reqid';
$not_merged_reqids = $output_dir||'/'||$date_minus;
$not_merged_reqids_cnt = $output_dir||'_cnt/'||$date_minus;

$strftime = ($timestamp) -> { return DateTime::MakeDatetime(DateTime::FromSeconds(cast($timestamp as UInt32))); };
$FormatDate = DateTime::Format('%Y-%m-%d');

$ts = ($reqid) -> {return cast(if(String::HasPrefix($reqid, 'MAIN_'),
                            substring($reqid, 5, 10),
                            substring($reqid, 0, 10)) as UInt64);};

$abt = (
    SELECT
        Yson::ConvertToString(key) as uid,
        String::SplitToList(_other['value'], '\t')[0] as reqid,
        String::Contains(_other['value'], 'wiz_avia_fraud=1') as fraud,
        String::Contains(_other['value'], 'wiz_avia_error=1') as error
    FROM $abt_log
);

$redirs = (select count(*) from $abt where not error);
$errors = (select count(*) from $abt where error);

$unmerged = (
    SELECT a.reqid as reqid,
        uid,
        fraud,
        error,
        len(a.reqid) as len_reqid,
        $FormatDate($strftime($ts(a.reqid))) != $date_minus as not_today_query,
        $strftime($ts(a.reqid)) as ts_time,
        $ts(a.reqid) as ts
    FROM $abt as a
    left only JOIN $abt_user_session as b ON a.reqid == Yson::ConvertToString(b.key) and a.uid == b._other['value']
);

INSERT INTO $not_merged_reqids_cnt WITH TRUNCATE
select $redirs as all_redirs,
    $errors as all_errors,
    count(*) as unmerged_cnt,
    count_if(len_reqid > 90 and not error) as unmerged_long_reqid_redirs,
    count_if(len_reqid > 90 and error) as unmerged_long_reqid_errors,
    count_if(not_today_query and not error) as unmerged_not_today_query_redirs,
    count_if(not_today_query and error) as unmerged_not_today_query_errors,
    count_if(fraud) as fraud_unmerged_redirs,
    count_if(error) as all_unmerged_errors,
    count_if(not error) as all_unmerged_redirs,
    count_if(not fraud and not not_today_query and len_reqid<=90 and not error) as good_but_unmerged_redirs,
    count_if(not fraud and not not_today_query and len_reqid<=90 and error) as good_but_unmerged_errors
from $unmerged;

INSERT INTO $not_merged_reqids WITH TRUNCATE
select *
from $unmerged
order by ts;
"""

QUERY_HOTELS = """
USE hahn;

DECLARE $date_minus AS String;
DECLARE $date AS String;
DECLARE $abt_log_dir AS String;
DECLARE $abt_user_session_dir AS String;
DECLARE $output_dir AS String;

$abt_log = $abt_log_dir||'/'||$date_minus;
$abt_user_session = $abt_user_session_dir||'/'||$date||'/reqid';
$not_merged_reqids = $output_dir||'/'||$date_minus;
$not_merged_reqids_cnt = $output_dir||'_cnt/'||$date_minus;

$strftime = ($timestamp) -> { return DateTime::MakeDatetime(DateTime::FromSeconds(cast($timestamp as UInt32))); };
$FormatDate = DateTime::Format('%Y-%m-%d');

$ts = ($reqid) -> {return cast(if(String::HasPrefix($reqid, 'MAIN_'),
                            substring($reqid, 5, 10),
                            substring($reqid, 0, 10)) as UInt64);};

$abt = (
    SELECT
        key as uid,
        String::SplitToList(value, '\t')[0] as reqid,
        String::Contains(value, 'profit_amount_rub') as profit
    FROM $abt_log
);

$redirs = (select count(*) from $abt where not profit);
$bookings = (select count(*) from $abt where profit);

$unmerged = (
    SELECT a.reqid as reqid,
        a.uid as uid,
        profit,
        len(a.reqid) as len_reqid,
        $FormatDate($strftime($ts(a.reqid))) != $date_minus as not_today_query,
        $ts(a.reqid) as ts
    FROM $abt as a
    left only JOIN $abt_user_session as b ON a.reqid == Yson::ConvertToString(b.key) and a.uid == b._other['value']
);

INSERT INTO $not_merged_reqids_cnt WITH TRUNCATE
select $redirs as all_redirs,
    $bookings as all_bookings,
    count(*) as unmerged_cnt,
    count_if(len_reqid > 90) as unmerged_long_reqid,
    count_if(profit and not_today_query) as unmerged_not_today_bookings,
    count_if(not profit and not_today_query) as unmerged_not_today_redirs,
    count_if(profit) as all_unmerged_bookings,
    count_if(not profit) as all_unmerged_redirs,
    count_if(profit and len_reqid <= 90 and not not_today_query) as good_but_unmerged_bookings,
    count_if(not profit and len_reqid <= 90 and not not_today_query) as good_but_unmerged_redirs
from $unmerged;

INSERT INTO $not_merged_reqids WITH TRUNCATE
select *
from $unmerged
order by ts;
"""


class SendAviaNotMergedUserSessionsToSolomon(AviaBaseTask):
    """
    Send Avia not merged redirects with abt-user-sessions to solomon.
    """
    _yt_client = None
    _yql_client = None

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('YT settings') as mr_block:
            yt_cluster = sdk2.parameters.String('YT cluster', default='hahn', required=True)
            yt_user = sdk2.parameters.String('YT user', required=True)
            yt_token_vault = sdk2.parameters.String('Token vault name', required=True, default='YT_TOKEN')

        with sdk2.parameters.Group('YQL settings') as yql_block:
            yql_token_vault = sdk2.parameters.String('YQL token vault name', required=True, default='YQL_TOKEN')

        with sdk2.parameters.Group('Solomon settings') as debug_settings:
            solomon_project = sdk2.parameters.String('Solomon project', required=True, default='avia')
            solomon_cluster = sdk2.parameters.String('Solomon cluster', required=True, default='yt')
            solomon_service = sdk2.parameters.String('Solomon service', required=True, default='redirects')

        with sdk2.parameters.Group('Settings') as date_block:
            left_date = sdk2.parameters.String('Start date (default yesterday)', required=False)
            right_date = sdk2.parameters.String('End date (default yesterday)', required=False)
            abt_log_dir = sdk2.parameters.String('Avia abt log path', required=True, default='//home/avia/abt')
            abt_user_session_dir = sdk2.parameters.String('Abt user session dir', required=True,
                                                          default='//home/abt-viewers/sessions_viewer/daily')
            output_dir = sdk2.parameters.String('Output dir', required=True,
                                                default='//home/avia/dev/kateov/not_merged_abt_reqids')
            for_hotels = sdk2.parameters.Bool('For hotels', default=False, required=True)

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            debug_run = sdk2.parameters.Bool('Debug run', default=False, required=True)

    class Requirements(sdk2.Requirements):
        # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

        environments = [sdk_environments.PipEnvironment('yandex-yt', version='0.10.8'),
                        sdk_environments.PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
                        sdk_environments.PipEnvironment('yql'),
                        sdk_environments.PipEnvironment('raven')]

    def _parse_date(self, s):
        if not s:
            return YESTERDAY
        try:
            return datetime.today() + timedelta(days=int(s))
        except ValueError:
            return datetime.strptime(s, '%Y-%m-%d')

    def create_table_with_yql_result(self, date):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder
        yql_client = self._get_yql_client()
        logging.info('self.Parameters.output_dir: %s', self.Parameters.output_dir)

        logging.info('%s', date.strftime('%Y%m%d'))
        logging.info('%s', date.strftime('%Y-%m-%d'))
        logging.info('%s', self.Parameters.abt_log_dir)
        logging.info('%s', self.Parameters.abt_user_session_dir)
        logging.info('%s', self.Parameters.output_dir)
        q = QUERY_HOTELS if self.Parameters.for_hotels else QUERY_AVIA
        logging.info('YQL query text: %s', q)
        r = yql_client.query(q, syntax_version=1)
        logging.info('YQL query: %s', r)
        r.run(parameters=ValueBuilder.build_json_map({
            '$date': ValueBuilder.make_string(date.strftime('%Y%m%d')),
            '$date_minus': ValueBuilder.make_string(date.strftime('%Y-%m-%d')),
            '$abt_log_dir': ValueBuilder.make_string(self.Parameters.abt_log_dir),
            '$abt_user_session_dir': ValueBuilder.make_string(self.Parameters.abt_user_session_dir),
            '$output_dir': ValueBuilder.make_string(self.Parameters.output_dir),
        }))
        logging.info('YQL Operation: %s', yqlh.get_yql_operation_url(r))
        r.wait_progress()

        if not r.is_success:
            logging.error('Operation failed. Status:', r.status)
            if r.errors:
                for error in r.errors:
                    logging.error(' - %s', str(error))
            raise SandboxTaskFailureError('YQL query failed')
        res_table = '{}_cnt/{}'.format(self.Parameters.output_dir, date.strftime('%Y-%m-%d'))
        logging.info('YQL query done in %s table', res_table)
        return res_table

    def _get_yt_client(self):
        if self._yt_client is None:
            import yt.wrapper as yt
            self._yt_client = yt.YtClient(
                proxy='hahn',
                token=self._get_token('yt'),
            )
            self._yt_json_format = yt.JsonFormat()

        return self._yt_client

    def _get_yql_client(self):
        if self._yql_client is None:
            from yql.api.v1.client import YqlClient
            self._yql_client = YqlClient(
                token=self._get_token('yql'),
            )

        return self._yql_client

    def _get_token(self, system):
        if system == 'yt':
            name = self.Parameters.yt_token_vault or 'YT_TOKEN'

        elif system == 'yql':
            name = self.Parameters.yql_token_vault or 'YQL_TOKEN'

        else:
            raise ValueError('Unknown system: {}'.format(system))

        return sdk2.Vault.data(self.Parameters.yt_user, name)

    def send_data_to_solomon(self, record, date):
        common_labels = {
            'project': self.Parameters.solomon_project,
            'cluster': self.Parameters.solomon_cluster,
            'service': self.Parameters.solomon_service,
        }
        logger.info(common_labels)
        sensors = [
            {
                'ts': UTC.localize(date).strftime('%Y-%m-%dT%H:%M:%SZ'),
                'labels': {'sensor': 'abt', 'counter': name},
                'value': cnt,
            }
            for name, cnt in record.items()
        ]
        logger.info(sensors)
        solomon.upload_to_solomon(common_labels, sensors)

    def process_date(self, date):
        logging.info('Process date: %s', date.strftime('%Y-%m-%d'))

        res_table = self.create_table_with_yql_result(date)
        yt = self._get_yt_client()

        record = yt.read_table(res_table, format=self._yt_json_format).next()
        logging.info('res_table=%s', res_table)
        logging.info(record)

        self.send_data_to_solomon(record, date)

    def on_execute(self):
        logs.configure_logging(logs.get_sentry_dsn(self))
        logging.info('Start')

        left_date = self._parse_date(self.Parameters.left_date)
        logging.info('Left date: %s', left_date.strftime('%Y-%m-%d'))
        right_date = self._parse_date(self.Parameters.right_date)
        logging.info('Right date: %s', right_date.strftime('%Y-%m-%d'))

        yt_client = self._get_yt_client()
        if not yt_client.exists(self.Parameters.output_dir):
            yt_client.create('map_node', self.Parameters.output_dir, recursive=True)
            logging.info('Outpus dir: %s created', self.Parameters.output_dir)
        else:
            logging.info('Outpus dir: %s exists', self.Parameters.output_dir)

        current_date = left_date
        while current_date <= right_date:
            self.process_date(current_date)
            current_date += timedelta(days=1)

        logging.info('End')
