# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.avia.base import AviaBaseTask

YESTERDAY = datetime.today() - timedelta(days=1)


class SendAviaRemBernsteinSignal(AviaBaseTask):
    """
    Send signal for avia-partner-booking-log.
    For cofe-metrics RASPTICKETS-14923
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            PipEnvironment('requests'),
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Date settings') as date_block:
            cluster = sdk2.parameters.String('YQL cluster', required=True, default='hahn')
            log_name = sdk2.parameters.String('Log name', required=True, default='avia_partner_booking_log')
            ready = sdk2.parameters.String('Table is ready (preliminary|main)', required=True, default='main')
            date = sdk2.parameters.String('Signal date (default yesterday, -2 - the day before yesterday)',
                                          required=False)

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            debug_run = sdk2.parameters.Bool('Debug run', default=False, required=True)

    def create_tag(self, date, cluster, log_name, ready):
        """
        :param date:
        :param cluster: hahn
        :param log_name: for example avia_partner_booking_log
        :param ready: preliminary|main
        :return:
        """
        return 'cluster={cluster}_cofe_travel_input_{log_name}_{ready}_daily_{date}'.format(
            cluster=cluster,
            log_name=log_name,
            ready=ready,
            date=date.strftime('%Y%m%d'),
        )

    def push_tag(self, tag):
        import requests
        return requests.get('http://rem-bernstein.n.yandex-team.ru/set_tag?tag=' + tag)

    def create_and_push_tag(self, date, cluster, log_name, ready):
        tag = self.create_tag(date, cluster, log_name, ready)
        logging.info('Tag: %s', tag)
        if not self.Parameters.debug_run:
            self.push_tag(tag)

    def on_execute(self):
        logging.info('Start')

        date = self._parse_date(self.Parameters.date).date()
        logging.info('Signal date: %s', date.strftime('%Y-%m-%d'))

        self.create_and_push_tag(date=date,
                                 cluster=self.Parameters.cluster,
                                 log_name=self.Parameters.log_name,
                                 ready=self.Parameters.ready)

        logging.info('End')

    def _parse_date(self, s):
        if not s:
            return YESTERDAY
        try:
            return datetime.today() + timedelta(days=int(s))
        except ValueError:
            return datetime.strptime(s, '%Y-%m-%d')
