# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib import yql_helpers as yqlh
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory

YESTERDAY = datetime.today() - timedelta(days=1)

DAILY_MODE = 'daily'
HALF_HOURLY_MODE = 'half-hourly'
DATETIME_FORMAT_BY_MODE = {DAILY_MODE: '%Y-%m-%d', HALF_HOURLY_MODE: '%Y-%m-%dT%H:%M:%S'}
TAG_DATE_FORMAT_BY_MODE = {DAILY_MODE: '%Y%m%d', HALF_HOURLY_MODE: '%Y%m%d%H%M'}


class SendTravelMetricsToABT(AviaBaseTask):
    """
    Send Travel wizards metrics to ABT.
    """

    _yt_client = None
    _yql_client = None

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('requests'),
            PipEnvironment('yql')
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Map reduce settings') as mr_block:
            mr_cluster = sdk2.parameters.String('MapReduce cluster', default='hahn', required=True)
            vaults_owner = sdk2.parameters.String('MapReduce user', required=True)
            mr_dir = sdk2.parameters.String('Directory', required=True, default='//home/avia/abt_wizard')
            tag_name = sdk2.parameters.String('Tag name', required=True, default='calc_avia_wizard_show_metrics')
            parameter_prefix = sdk2.parameters.String('Parameter prefix', required=True, default='wiz_avia')
            yql_title = sdk2.parameters.String('YQL title',
                                               required=True,
                                               default='[YQL] Generating Avia wizard show metrics')
            yt_token_vault_name = sdk2.parameters.String('YT Token vault name', required=True, default='YT_TOKEN')

        with sdk2.parameters.Group('YQL settings') as yql_block:
            yql_token_vault_name = sdk2.parameters.String('YQL token vault name', required=True, default='YQL_TOKEN')
            query = sdk2.parameters.String("YQL Query", multiline=True, required=False)

        with sdk2.parameters.Group('Date settings') as date_block:
            with sdk2.parameters.RadioGroup('Mode') as mode:
                mode.values[DAILY_MODE] = mode.Value(DAILY_MODE, default=True)
                mode.values[HALF_HOURLY_MODE] = mode.Value(HALF_HOURLY_MODE)

                with mode.value[HALF_HOURLY_MODE]:
                    left_datetime = sdk2.parameters.String(
                        'Start datetime',
                        description=DATETIME_FORMAT_BY_MODE[HALF_HOURLY_MODE],
                        required=False,
                    )
                    right_datetime = sdk2.parameters.String(
                        'End datetime',
                        description=DATETIME_FORMAT_BY_MODE[HALF_HOURLY_MODE],
                        required=False,
                    )

                with mode.value[DAILY_MODE]:
                    left_date = sdk2.parameters.String('Start date (default yesterday)', required=False)
                    right_date = sdk2.parameters.String('End date (default yesterday)', required=False)

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            debug_run = sdk2.parameters.Bool('Debug run', default=False, required=True)

    @property
    def yt_client(self):
        if self._yt_client is None:
            self._yt_client = YtClientFactory.create(
                proxy=self.Parameters.mr_cluster,
                token=sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yt_token_vault_name),
            )

        return self._yt_client

    @property
    def yql_client(self):
        if self._yql_client is None:
            from yql.api.v1.client import YqlClient
            self._yql_client = YqlClient(
                token=sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yql_token_vault_name),
                db=self.Parameters.mr_cluster,
            )

        return self._yql_client

    def create_tag(self, date, datetime_format, cluster='hahn'):
        return 'cluster={cluster}_{tag_name}_{date}'.format(
            cluster=cluster,
            tag_name=self.Parameters.tag_name,
            date=date.strftime(datetime_format),
        )

    def set_tag(self, tag):
        import requests
        return requests.get('http://rem-bernstein.n.yandex-team.ru/set_tag?tag=' + tag)

    def create_and_set_tag(self, date, mode):
        tag = self.create_tag(date, TAG_DATE_FORMAT_BY_MODE[mode], self.Parameters.mr_cluster)
        logging.info('Tag: %s', tag)
        if not self.Parameters.debug_run:
            self.set_tag(tag)

    def process_part(self, date, output_path, mode):
        logging.info('Process date: %s', date.strftime(DATETIME_FORMAT_BY_MODE[mode]))

        output_table = os.path.join(output_path, date.strftime(DATETIME_FORMAT_BY_MODE[mode]))
        self.create_temp_table_with_abt_metrics(date, output_table, mode)
        self.create_and_set_tag(date, mode)

    def create_temp_table_with_abt_metrics(self, date, output_table, mode):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        r = self.yql_client.query(
            self.Parameters.query,
            syntax_version=1,
            title=self.Parameters.yql_title,
        ).run(parameters=ValueBuilder.build_json_map({
            '$OutputPath': ValueBuilder.make_string(output_table),
            '$Date': ValueBuilder.make_date(date) if mode == DAILY_MODE else ValueBuilder.make_datetime(date),
            '$parameter_prefix': ValueBuilder.make_string(self.Parameters.parameter_prefix)
        }))

        logging.info('YQL Operation: %s', r.share_url)
        r.wait_progress()

        if not r.is_success:
            yqlh.log_errors(r, logging)
            raise SandboxTaskFailureError('YQL query failed')

        logging.info('YQL query done in %s table', output_table)

    @staticmethod
    def _parse_datetime(s, mode):
        if not s:
            if mode == DAILY_MODE:
                return YESTERDAY
            today = datetime.today() - timedelta(hours=2)
            # (2021-03-08 22:16:01) -> (2021-03-08 22:00:00)
            # (2021-03-08 22:30:12) -> (2021-03-08 22:30:00)
            return datetime(*today.timetuple()[:4]) + timedelta(minutes=30 * int(today.minute >= 30))
        try:
            return datetime.today() + timedelta(days=int(s))
        except ValueError:
            return datetime.strptime(s, TAG_DATE_FORMAT_BY_MODE[mode])

    def on_execute(self):
        logging.info('Start')

        if not self.yt_client.exists(self.Parameters.mr_dir):
            self.yt_client.create('map_node', self.Parameters.mr_dir, recursive=True)

        mode = self.Parameters.mode

        if mode not in {DAILY_MODE, HALF_HOURLY_MODE}:
            raise Exception('Unknown run mode: %s', self.Parameters.mode)

        if mode == DAILY_MODE:
            start = self.Parameters.left_date
            end = self.Parameters.right_date
        else:
            start = self.Parameters.left_datetime
            end = self.Parameters.right_datetime

        start = self._parse_datetime(start, mode)
        end = self._parse_datetime(end, mode)

        logging.info('Start date: %s', start.strftime(DATETIME_FORMAT_BY_MODE[mode]))
        logging.info('End date: %s', end.strftime(DATETIME_FORMAT_BY_MODE[mode]))

        step_by_mode = {DAILY_MODE: timedelta(days=1), HALF_HOURLY_MODE: timedelta(minutes=30)}
        parts = self._generate_parts(start, end, step_by_mode[mode])
        for part in parts:
            self.process_part(part, self.Parameters.mr_dir, mode)

        logging.info('End')

    def _generate_parts(self, start, end, step):
        current_date = start
        while current_date <= end:
            yield current_date
            current_date += step
