# coding=utf-8
from __future__ import unicode_literals

import logging

from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.datetime_helpers import get_utc_now, _dt_to_string
from sandbox.projects.common import binary_task, solomon
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

logger = logging.getLogger(__name__)
YDB_TABLES = (
    'organic_results_preproduction',
    'organic_results_production',
    'redirect_data',
    'results',
    'results_expiration_queue0',
    'results_expiration_queue1',
    'results_expiration_queue2',
    'results_expiration_queue3',
    'results_expiration_queue4',
    'results_expiration_queue5',
    'results_expiration_queue6',
    'results_expiration_queue7',
    'results_expiration_queue8',
    'results_expiration_queue9',
    'wizard_results',
    'wizard_results_expiration_queue0',
    'wizard_results_expiration_queue1',
    'wizard_results_expiration_queue2',
    'wizard_results_expiration_queue3',
    'wizard_results_expiration_queue4',
    'wizard_results_expiration_queue5',
    'wizard_results_expiration_queue6',
    'wizard_results_expiration_queue7',
    'wizard_results_expiration_queue8',
    'wizard_results_expiration_queue9',
    'wizard_results_experimental',
    'wizard_results_experimental_expiration_queue0',
    'wizard_results_experimental_expiration_queue1',
    'wizard_results_experimental_expiration_queue2',
    'wizard_results_experimental_expiration_queue3',
    'wizard_results_experimental_expiration_queue4',
    'wizard_results_experimental_expiration_queue5',
    'wizard_results_experimental_expiration_queue6',
    'wizard_results_experimental_expiration_queue7',
    'wizard_results_experimental_expiration_queue8',
    'wizard_results_experimental_expiration_queue9',
)


class AviaSendYdbTablesRowCountToSolomon(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    """Send count(*) from YDB_TABLES to solomon"""
    _yql_client = None

    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('YQL settings') as yql_settings:
            yav_yql_token = sdk2.parameters.YavSecret(
                "Yav-secret с YQL-токеном робота",
                default='sec-01e4q88q2jcz84c8dtwrm29z3r'
            )

        with sdk2.parameters.Group('Solomon settings') as solomon_settings:
            solomon_project = sdk2.parameters.String('Solomon project', required=True, default='avia')
            solomon_cluster = sdk2.parameters.String('Solomon cluster', required=True, default='ydb')
            solomon_service = sdk2.parameters.String('Solomon service', required=True, default='metrics')

        with sdk2.parameters.Group('YDB settings') as ydb_settings:
            ydb_database = sdk2.parameters.String(
                'YDB database',
                required=True,
                default='/ru/ticket/production/search_results'
            )

        with sdk2.parameters.Group('YDB ClickHouse settings') as ydb_click_house:
            ydb_click_house_balancer = sdk2.parameters.String(
                'YDB ClickHouse balancer',
                required=True,
                default='http://ydb-ru-prestable-ch.yandex.net:8123/'
            )

    def on_execute(self):
        super(AviaSendYdbTablesRowCountToSolomon, self).on_execute()
        start_dt = get_utc_now()
        tables_row_count = self.get_row_count()
        self.send_data_to_solomon(tables_row_count, start_dt)

    def get_row_count(self):
        results = []

        for table in YDB_TABLES:
            query = self.get_table_row_count_query(table)
            response = self.send_request(query)

            if response:
                logger.info('YDB over ClickHouse response %s', response.content)
                try:
                    row_count = int(response.content)
                except ValueError:
                    row_count = None

                results.append((table, row_count))
                logging.info('YDB over ClickHouse query done for %s table. Row count %r', table, row_count)
        if not results:
            raise SandboxTaskFailureError('No results from YDB over ClickHouse')

        return results

    def send_data_to_solomon(self, tables_row_count, start_dt):
        shard_labels = {
            'project': self.Parameters.solomon_project,
            'cluster': self.Parameters.solomon_cluster,
            'service': self.Parameters.solomon_service,
        }
        logger.info(shard_labels)
        sensors = []

        for table, row_count in tables_row_count:
            sensors.append(
                {
                    'ts': _dt_to_string(start_dt) + 'Z',
                    'labels': {'sensor': 'row_count', 'table': table},
                    'value': row_count,
                }
            )

        logger.info('Sending sensors to solomon. %r', sensors)
        solomon.push_to_solomon_v2(self.solomon_token, shard_labels, sensors, common_labels=())

    def get_table_row_count_query(self, table):
        query = """SELECT count(*) AS cnt FROM ydbTable('ru', '{ydb_database}', '{table}')""".format(
            ydb_database=self.Parameters.ydb_database,
            table=table,
        )

        return query

    def send_request(self, query):
        import requests
        from requests.auth import HTTPBasicAuth

        logging.info('ClickHouse over YDB Operation: %s', query)
        yql_token = self.Parameters.yav_yql_token.data()['token']
        try:
            response = requests.post(
                self.Parameters.ydb_click_house_balancer,
                data=query,
                auth=HTTPBasicAuth('default', yql_token)
            )
        except Exception as exc:
            logger.warning('No results with exception %r', exc)
            return

        if not response.ok:
            logger.warning('Bad status code %r', response.status_code)
            return

        return response
