# coding=utf-8
import logging
import traceback
from datetime import datetime

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.shared_flights.amadeus_parser_resources import resources
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

PRODUCTION_ENVIRONMENT = 'production'


class AviaSharedFlightsAmadeusParserTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    """ Parses flights data files from Amadeus """

    class Requirements(sdk2.Task.Requirements):
        # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Import parameters') as import_block:
            login = sdk2.parameters.String(
                'Amadeus sftp login',
                required=False,
                default_value='vasbur@yandex-team.ru',
            )
            password = sdk2.parameters.YavSecret(
                'Amadeus sftp password',
                default='sec-01f1fmnx646by5g3taafjaxzr8',
                required=True,
            )
            environment = sdk2.parameters.String(
                'Environment (testing or production)',
                required=True,
                default_value='testing',
            )

    def on_prepare(self):
        configure_logging(
            sentry_dsn=get_sentry_dsn(self)
        )
        self._logger = logging.getLogger(__name__)

        super(AviaSharedFlightsAmadeusParserTask, self).on_prepare()

    def on_execute(self):
        import shutil
        from travel.avia.library.python.proxy_pool.deploy_proxy_pool import DeployProxyPool
        from travel.avia.shared_flights.tasks.amadeus_parser.amadeus_fetcher import AmadeusFetcher

        ftp_password = self.Parameters.password.data()['pwd']

        parsed_at = datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S')

        existing_resources = []
        if self.Parameters.environment == PRODUCTION_ENVIRONMENT:
            existing_resources = list(
                resources.AviaSharedFlightsLastUpdatedAmadeusProductionResource.find(state=ctr.State.READY).limit(2))
        else:
            existing_resources = list(
                resources.AviaSharedFlightsLastUpdatedAmadeusTestingResource.find(state=ctr.State.READY).limit(2))

        if len(existing_resources) > 1:
            raise SandboxTaskFailureError(
                'Too many AviaSharedFlightsLastUpdatedAmadeusResource instances found. '
                'Please manually delete all except one.'
            )
        elif len(existing_resources) == 1:
            resource = existing_resources[0]
        else:
            resource = None
            if self.Parameters.environment == PRODUCTION_ENVIRONMENT:
                resource = resources.AviaSharedFlightsLastUpdatedAmadeusProductionResource(self, 'Latest processed date', '')
            else:
                resource = resources.AviaSharedFlightsLastUpdatedAmadeusTestingResource(self, 'Latest processed date', '')
            # Create dummy file, as sandbox doesn't like it when the newly created resource has no files
            resource_data = sdk2.ResourceData(resource)
            resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
            last_parsed_path = resource_data.path.joinpath('last-parsed')
            last_parsed_path.write_bytes('Tracking resource, created at {}'.format(parsed_at))
            resource_data.ready()

        proxy_pool = DeployProxyPool(
            environment=self.Parameters.environment,
            datacenter='sas',
            login=None,
            password=None,
            logger=self._logger,
        )

        amadeus_fetcher = AmadeusFetcher(
            hostname=proxy_pool.get_proxy().get_ftp_host(),
            username=self.Parameters.login,
            password=ftp_password,
            logger=self._logger,
        )
        try:
            result = amadeus_fetcher.fetch(resource.last_parsed_date)

            if result is not None:
                # create output resource
                if self.Parameters.environment == PRODUCTION_ENVIRONMENT:
                    output_resource = resources.AviaSharedFlightsAmadeusProductionResource(
                        self,
                        "Parsed Amadeus files",
                        "references",
                    )
                else:
                    output_resource = resources.AviaSharedFlightsAmadeusTestingResource(
                        self,
                        "Parsed Amadeus files",
                        "references",
                    )
                output_resource.created_at = parsed_at
                output_resource.source_data_date = result.data_file_timestamp
                output_resource_data = sdk2.ResourceData(output_resource)
                output_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)

                # copy temp files into the resource files
                flight_bases_path = output_resource_data.path.joinpath("flight_bases.bin")
                shutil.copyfile(result.flight_bases_file, flight_bases_path.absolute().as_posix())

                flight_patterns_path = output_resource_data.path.joinpath("flight_patterns.bin")
                shutil.copyfile(result.flight_patterns_file, flight_patterns_path.absolute().as_posix())

                amadeus_codeshares_path = output_resource_data.path.joinpath("codeshares.bin")
                shutil.copyfile(result.codeshares_file, amadeus_codeshares_path.absolute().as_posix())

                self._logger.info('Data files are processed, updating the tracking record resource.')

                output_resource_data.ready()
                resource.last_parsed_date = result.data_file_timestamp
                resource.parsed_at = parsed_at
                resource.last_parsed_resource_id = output_resource.id

            self._logger.info('Task has been completed.')
        except Exception as e:
            raise SandboxTaskFailureError('Unable to fetch Amadeus files. \n {} \n {}'.format(traceback.format_exc(), e))
