# -*- encoding: utf-8 -*-
import logging
import os

from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
import sandbox.common.types.misc as ctm


class AviaSharedFlightsDataImporterTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 12292

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Parameters') as executable_params:
            script_params = sdk2.parameters.List(
                'Parameters',
                description='--nodaemon',
                default=['--nodaemon'],
            )

        with sdk2.parameters.Group('Environment') as environment_settings:
            environment_variables = sdk2.parameters.Dict('Environment variables')

        with sdk2.parameters.Group('Secrets') as secrets_settings:
            pg_password_secret = sdk2.parameters.YavSecret(
                'Yav secret with PG password for flight storage',
                required=True,
                default='sec-01ejr2whpm3j9hscf7y6196gg2',
            )

    def on_prepare(self):
        configure_logging(
            sentry_dsn=get_sentry_dsn(self)
        )
        self._logger = logging.getLogger(__name__)

        super(AviaSharedFlightsDataImporterTask, self).on_prepare()

    def on_execute(self):
        logging.info('Execution started')
        super(AviaSharedFlightsDataImporterTask, self).on_execute()

        logging.info('Creating env variables')
        environment_variables = self.Parameters.environment_variables
        environment_variables['AVIA_PGAAS_PASSWORD'] = self.get_secret_value(self.Parameters.pg_password_secret, 'password')

        for key, value in environment_variables.items():
            os.environ[key]=value

        try:
            from travel.avia.shared_flights.data_importer.flights_data_importer import run_import

            self._logger.info('Launching data importer')
            run_import(self.Parameters.script_params)
            self._logger.info('Task has been completed.')
        except Exception as e:
            raise SandboxTaskFailureError('Unable to import flights data. {}'.format(e))
