# -*- encoding: utf-8 -*-
import logging
import os

from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.shared_flights.diff_builder_resources import resources
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
import sandbox.common.types.misc as ctm

PRODUCTION_ENVIRONMENT = 'production'


class AviaSharedFlightsDiffBuilderTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 12292

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Parameters') as executable_params:
            script_params = sdk2.parameters.List(
                'Parameters',
                description='--force if needed',
                default=[],
            )

        with sdk2.parameters.Group('Environment') as environment_settings:
            environment_variables = sdk2.parameters.Dict('Environment variables')

        auto = sdk2.parameters.Bool('Auto')
        with auto.value[True]:
            with sdk2.parameters.RadioGroup('Environment') as environment:
                environment.values.TESTING = environment.Value('testing', default=True)
                environment.values.PRODUCTION = environment.Value('production')

        with sdk2.parameters.Group('Secrets') as secrets_settings:
            pg_password_secret = sdk2.parameters.YavSecret(
                'Yav secret with PG password for flight storage',
                required=True,
                default='sec-01ejr2whpm3j9hscf7y6196gg2',
            )
            mds_access_key_id = sdk2.parameters.YavSecret(
                'Yav secret with AVIA_MDS_ACCESS_KEY_ID',
                required=True,
                default='sec-01dsexxyfkmr6spccdc1q385wp',
            )
            mds_access_key_secret = sdk2.parameters.YavSecret(
                'Yav secret with AVIA_MDS_ACCESS_KEY_SECRET',
                required=True,
                default='sec-01dsexxyfkmr6spccdc1q385wp',
            )

    def on_prepare(self):
        configure_logging(
            sentry_dsn=get_sentry_dsn(self)
        )
        self._logger = logging.getLogger(__name__)

        super(AviaSharedFlightsDiffBuilderTask, self).on_prepare()

    def on_execute(self):
        logging.info('Execution started')
        super(AviaSharedFlightsDiffBuilderTask, self).on_execute()

        logging.info('Creating env variables')
        environment_variables = self.Parameters.environment_variables
        environment_variables['AVIA_PGAAS_PASSWORD'] = self.get_secret_value(self.Parameters.pg_password_secret, 'password')
        environment_variables['AVIA_MDS_ACCESS_KEY_ID'] = self.get_secret_value(self.Parameters.mds_access_key_id, 'aws_access_key_id')
        environment_variables['AVIA_MDS_ACCESS_KEY_SECRET'] = self.get_secret_value(
            self.Parameters.mds_access_key_secret,
            'aws_secret_access_key',
        )

        for key, value in environment_variables.items():
            os.environ[key] = value

        try:
            from travel.avia.shared_flights.diff_builder.start_app import run_build_snapshot

            self._logger.info('Launching diff builder')

            output_resource = self.create_output_resource(self.Parameters.environment)
            run_build_snapshot(self.Parameters.script_params, output_resource.path.absolute().as_posix())

            output_resource.ready()
            self._logger.info('Task has been completed.')
        except Exception as e:
            raise SandboxTaskFailureError('Unable to build fligths data. {}'.format(e))

    def create_output_resource(self, environment):
        resource_file_name = 'snapshot.zip'
        if environment == PRODUCTION_ENVIRONMENT:
            output_resource = resources.AviaSharedFlightsDiffBuilderSnapshotProductionResource(
                self, 'Dump of the year-long schedules', resource_file_name)
        else:
            output_resource = resources.AviaSharedFlightsDiffBuilderSnapshotTestingResource(
                self, 'Dump of the year-long schedules', resource_file_name)

        return sdk2.ResourceData(output_resource)
