# -*- encoding: utf-8 -*-
import logging
import os

from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.shared_flights.flying_time_resources import resources
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
import sandbox.common.types.misc as ctm


PRODUCTION_ENVIRONMENT = 'production'


class AviaSharedFlightsFlyingTimeTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 12292

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Parameters') as executable_params:
            environment = sdk2.parameters.String(
                'Environment (testing or production)',
                required=True,
                default_value='testing',
            )

        with sdk2.parameters.Group('Environment') as environment_settings:
            environment_variables = sdk2.parameters.Dict('Environment variables')

        with sdk2.parameters.Group('Secrets') as secrets_settings:
            pg_password_secret = sdk2.parameters.YavSecret(
                'Yav secret with PG password for the shared-flights database',
                required=True,
                default='sec-01e84cae2rjpe3pvwjrt9c0phb',
            )

    def on_prepare(self):
        configure_logging(
            sentry_dsn=get_sentry_dsn(self)
        )
        self._logger = logging.getLogger(__name__)

        super(AviaSharedFlightsFlyingTimeTask, self).on_prepare()

    def on_execute(self):
        logging.info('Execution started')
        super(AviaSharedFlightsFlyingTimeTask, self).on_execute()

        logging.info('Creating env variables')
        environment_variables = self.Parameters.environment_variables
        environment_variables['AVIA_PGAAS_PASSWORD'] = self.get_secret_value(self.Parameters.pg_password_secret, 'password')

        for key, value in environment_variables.items():
            os.environ[key]=value

        try:
            from travel.avia.shared_flights.tasks.flying_time.flying_time import FlyingTimeTool

            self._logger.info('Launching flying time dumper')
            flying_time_tool = FlyingTimeTool(self.Parameters.environment, 'binary', '', 0)
            binary_data = flying_time_tool.run()

            if not binary_data:
                raise SandboxTaskFailureError('Dumper has returned no flying times.')

            # create output resource
            output_resource = None
            resource_file_name = 'flying_times.bin'
            if self.Parameters.environment == PRODUCTION_ENVIRONMENT:
                output_resource = resources.AviaSharedFlightsFlyingTimeProductionResource(
                    self, 'Minimal flying times between settlements', resource_file_name)
            else:
                output_resource = resources.AviaSharedFlightsFlyingTimeTestingResource(
                    self, 'Minimal flying times between settlements', resource_file_name)

            # copy results into the resource files
            output_resource_data = sdk2.ResourceData(output_resource)
            with open(output_resource_data.path.absolute().as_posix(), 'wb') as output_file:
                output_file.write(binary_data.getvalue())
            self._logger.info('Settlement to settlement flying times have been dumped.')

            output_resource_data.ready()

            self._logger.info('Task has been completed.')
        except Exception as e:
            raise SandboxTaskFailureError('Unable to dump flying times. {}'.format(e))
