# -*- encoding: utf-8 -*-
import logging

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.shared_flights.quality_tool_resources import resources
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

PRODUCTION_ENVIRONMENT = 'production'


class AviaSharedFlightsQualityToolTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Common settings') as common_settings:
            pg_password = sdk2.parameters.YavSecret(
                'Yav secret with Postgres password',
                required=True,
                default='sec-01e84cae2rjpe3pvwjrt9c0phb',
            )
            environment = sdk2.parameters.String(
                'Shared-flights environment',
                required=True,
                default_value='testing',
            )
            stations_from = sdk2.parameters.String(
                'Stations from (iata)',
                required=False,
                default='',
            )
            stations_to = sdk2.parameters.String(
                'Stations to (iata)',
                required=False,
                default='',
            )
            debug = sdk2.parameters.Bool(
                'Log at debug level',
                required=True,
                default_value=False,
            )

    def on_prepare(self):
        self._logger = logging.getLogger(__name__)

        super(AviaSharedFlightsQualityToolTask, self).on_prepare()

    def on_execute(self):
        from travel.avia.shared_flights.tasks.quality_tool.quality_tool import QualityTool

        super(AviaSharedFlightsQualityToolTask, self).on_execute()

        pg_password = self.Parameters.pg_password.data()['password']

        try:
            self._logger.info('Launching quality tool.')
            quality_tool = QualityTool(
                self.Parameters.environment,
                self.Parameters.stations_from,
                self.Parameters.stations_to,
                pg_password,
                self.Parameters.debug,
            )
            results = quality_tool.run()

            if results:
                self._logger.info('Got results, saving them as a resource.')

                # create output resource
                output_resource = None
                if self.Parameters.environment == PRODUCTION_ENVIRONMENT:
                    output_resource = resources.AviaSharedFlightsQualityToolProductionResource(
                        self, "Quality tool comparison results", "references")
                else:
                    output_resource = resources.AviaSharedFlightsQualityToolTestingResource(
                        self, "Quality tool comparison results", "references")

                output_resource_data = sdk2.ResourceData(output_resource)
                output_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)

                # copy results into the resource files
                output_path = output_resource_data.path.joinpath("results.csv")
                with open(output_path.absolute().as_posix(), 'wb') as output_file:
                    output_file.write(results)
                self._logger.info('Comparison results have been processed.')

                output_resource_data.ready()

            self._logger.info('Task has been completed.')
        except Exception as e:
            raise SandboxTaskFailureError('Unable to run quality tool. {}'.format(e))
