# coding=utf-8
from __future__ import unicode_literals

import logging
from datetime import datetime

from six import iteritems

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.datetime_helpers import (_dt_to_string, get_utc_now)
from sandbox.projects.avia.shared_flights.amadeus_parser_resources.resources import \
    AviaSharedFlightsAmadeusProductionResource, AviaSharedFlightsAmadeusTestingResource
from sandbox.projects.avia.shared_flights.sirena_parser_resources.resources import \
    AviaSharedFlightsSirenaProductionResource, AviaSharedFlightsSirenaTestingResource
from sandbox.projects.avia.shared_flights.ssim_parser_resources.resources import \
    AviaSharedFlightsInnovataProductionResource, AviaSharedFlightsInnovataTestingResource
from sandbox.projects.common import binary_task, solomon

logger = logging.getLogger(__name__)


class ResourceMetricsCollector(object):
    def __init__(self, resource):
        """
        :param Type[sdk2.Resource] resource:
        """
        self.resource = resource

    def get_latest_resource(self):
        return self.resource.find(state=ctr.State.READY).first()

    def _get_metrics(self, latest_resource):
        # type: (sdk2.Resource)->dict
        raise NotImplementedError()

    def get_metrics(self):
        for k, v in iteritems(self._get_metrics(self.get_latest_resource())):
            if v is not None:
                yield (
                    {
                        'sensor': k,
                        'resource_name': self.resource.name,
                    }, v
                )


class InnovataResourceMetricsCollector(ResourceMetricsCollector):
    def _get_metrics(self, latest_resource):  # type: (AviaSharedFlightsInnovataResource)->dict
        default_metrics = {'available': 0, 'time_since_created': None, 'time_since_source_date': None}
        if latest_resource:
            now_utc_aware = datetime.utcnow()

            created_at_utc_aware = datetime.strptime(
                latest_resource.created_at, '%Y-%m-%d %H:%M:%S',
            )
            default_metrics['time_since_created'] = (now_utc_aware - created_at_utc_aware).total_seconds()

            source_utc_aware = datetime.strptime(
                latest_resource.source_data_date, '%Y-%m-%d %H:%M:%S',
            )
            default_metrics['time_since_source_date'] = (now_utc_aware - source_utc_aware).total_seconds()

            default_metrics['size'] = latest_resource.size

            default_metrics['available'] = 1
        return default_metrics


class SirenaResourceMetricsCollector(ResourceMetricsCollector):
    def _get_metrics(self, latest_resource):  # type: (AviaSharedFlightsSirenaResource)->dict
        default_metrics = {'available': 0, 'time_since_created': None}
        if latest_resource:
            now_utc_aware = datetime.utcnow()

            created_at_utc_aware = datetime.strptime(
                latest_resource.created_at, '%Y-%m-%d %H:%M:%S',
            )
            default_metrics['time_since_created'] = (now_utc_aware - created_at_utc_aware).total_seconds()

            default_metrics['size'] = latest_resource.size

            default_metrics['available'] = 1
        return default_metrics


class AmadeusResourceMetricsCollector(ResourceMetricsCollector):
    def _get_metrics(self, latest_resource):
        # type: (AviaSharedFlightsAmadeusResource)->dict
        default_metrics = {'available': 0, 'time_since_created': None, 'time_since_source_date': None}
        if latest_resource:
            now_utc_aware = datetime.utcnow()

            created_at_utc_aware = datetime.strptime(
                latest_resource.created_at, '%Y-%m-%d %H:%M:%S',
            )
            default_metrics['time_since_created'] = (now_utc_aware - created_at_utc_aware).total_seconds()

            source_utc_aware = datetime.strptime(
                latest_resource.source_data_date, '%Y-%m-%d %H:%M:%S',
            )
            default_metrics['time_since_source_date'] = (now_utc_aware - source_utc_aware).total_seconds()

            default_metrics['size'] = latest_resource.size

            default_metrics['available'] = 1
        return default_metrics


_TESTING_ = 'testing'
_PRODUCTION_ = 'production'

RESOURCES_TO_TRACK = {
    _TESTING_: {
        AviaSharedFlightsAmadeusTestingResource: AmadeusResourceMetricsCollector,
        AviaSharedFlightsInnovataTestingResource: InnovataResourceMetricsCollector,
        AviaSharedFlightsSirenaTestingResource: SirenaResourceMetricsCollector,
    },
    _PRODUCTION_: {
        AviaSharedFlightsAmadeusProductionResource: AmadeusResourceMetricsCollector,
        AviaSharedFlightsInnovataProductionResource: InnovataResourceMetricsCollector,
        AviaSharedFlightsSirenaProductionResource: SirenaResourceMetricsCollector,
    },
}


class AviaSharedFlightsSendResourceStatsToSolomon(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    """
    Мониторинг живости парсеров расписаний
    """

    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Solomon settings') as solomon_settings:
            solomon_project = sdk2.parameters.String('Solomon project', required=True, default='avia')
            solomon_cluster = sdk2.parameters.String('Solomon cluster', required=True, default='shared-flights')
            solomon_service = sdk2.parameters.String('Solomon service', required=True, default='app')
            solomon_component = sdk2.parameters.String('Solomon component', required=True, default='schedule-parser')

        with sdk2.parameters.RadioGroup('Environment') as environment_group:
            environment_group.values.testing = environment_group.Value(_TESTING_, default=True)
            environment_group.values.production = environment_group.Value(_PRODUCTION_)

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        super(AviaSharedFlightsSendResourceStatsToSolomon, self).on_execute()
        metrics = []
        for resource_type, metrics_collector in iteritems(
            RESOURCES_TO_TRACK[self.Parameters.environment_group],
        ):
            resource_metric = metrics_collector(resource_type).get_metrics()
            for sensor_data in resource_metric:
                metrics.append(sensor_data)
        self.send_data_to_solomon(metrics, get_utc_now())

    def send_data_to_solomon(self, metrics, timestamp):
        solomon_shard = {
            'project': self.Parameters.solomon_project,
            'cluster': '_'.join([self.Parameters.solomon_cluster, self.Parameters.environment_group]),
            'service': self.Parameters.solomon_service,
        }
        logger.info(solomon_shard)
        common_labels = {
            'component': self.Parameters.solomon_component,
        }
        sensors = []

        for labels, value in metrics:
            sensors.append(
                {
                    'ts': _dt_to_string(timestamp) + 'Z',
                    'labels': labels,
                    'value': value,
                }
            )

        logger.info('Sending sensors to solomon. %r', sensors)
        solomon.push_to_solomon_v2(self.solomon_token, solomon_shard, sensors, common_labels)
        logger.info('Done sending sensors to solomon')
