# coding=utf-8
from __future__ import unicode_literals

import logging
import traceback
from datetime import datetime

import sandbox.common.types.misc as ctm
import sandbox.common.types.notification as ctn
from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.shared_flights.sirena_parser_resources import resources
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

PRODUCTION_ENVIRONMENT = 'production'


class AviaSharedFlightsSirenaParserTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    """ Parses schedule files from Sirena """

    class Requirements(sdk2.Task.Requirements):
        # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task does not use any shared caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Import parameters') as import_block:
            host = sdk2.parameters.String(
                'Sirena socket host (leave blank for default value)',
                required=False,
                default_value='',
            )
            port_name = sdk2.parameters.String(
                'Port name (study or production)',
                required=False,
                default_value='study',
            )
            client_id = sdk2.parameters.Integer(
                'Our client ID at Sirena',
                required=False,
                default_value=7724,
            )
            carrier_codes = sdk2.parameters.String(
                'Carrier codes to parse',
                required=True,
                default_value='',
            )
            vaults_owner = sdk2.parameters.String(
                'SSIM files ftp password vault entry owner',
                required=True,
                default_value='AVIA',
            )
            environment = sdk2.parameters.String(
                'Environment (testing or production)',
                required=True,
                default_value='testing',
            )
            new_transport_models_notification_recipients = sdk2.parameters.List(
                'New transport models notification recipients',
                required=False,
                default=[],
                value_type=sdk2.parameters.String,
            )

    def on_prepare(self):
        configure_logging(
            sentry_dsn=get_sentry_dsn(self)
        )
        self._logger = logging.getLogger(__name__)

        super(AviaSharedFlightsSirenaParserTask, self).on_prepare()

    def on_execute(self):
        from travel.avia.shared_flights.tasks.sirena_parser.sirena_fetcher import SirenaFetcher

        parsed_at = datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S')

        sirena_fetcher = SirenaFetcher(
            host=self.Parameters.host,
            port_name=self.Parameters.port_name,
            client_id=self.Parameters.client_id,
            carrier_codes=self.Parameters.carrier_codes,
            oauth_token=None,
        )
        try:
            mem_zip_file, new_transport_models = sirena_fetcher.fetch()
            self._logger.info('Done with Sirena fetching.')

            if mem_zip_file is not None:
                # create output resource
                output_resource = None
                if self.Parameters.environment == PRODUCTION_ENVIRONMENT:
                    output_resource = resources.AviaSharedFlightsSirenaProductionResource(
                        self,
                        "Parsed sirena files",
                        "references",
                    )
                else:
                    output_resource = resources.AviaSharedFlightsSirenaTestingResource(
                        self,
                        "Parsed sirena files",
                        "references",
                    )

                output_resource.created_at = parsed_at
                output_resource_data = sdk2.ResourceData(output_resource)
                output_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)

                # copy temp files into the resource files
                output_path = output_resource_data.path.joinpath("sirena.zip")
                with open(output_path.absolute().as_posix(), 'wb') as output_file:
                    output_file.write(mem_zip_file.getvalue())
                self._logger.info('Data files have been processed.')

                output_resource_data.ready()

            recipients = self.Parameters.new_transport_models_notification_recipients
            if recipients and new_transport_models:
                self._logger.info(
                    'Reporting new transport models to recipients: %s',
                    recipients,
                )
                message_body = '\n'.join(new_transport_models)
                self._logger.info('Message body: %s', message_body)
                self.server.notification(
                    subject='Новые модели транспорта из Сирены от {}'.format(datetime.today().strftime('%Y-%m-%d')),
                    body=message_body,
                    recipients=recipients,
                    transport=ctn.Transport.EMAIL,
                    type='plain',
                )

            self._logger.info('Task has been completed.')
        except Exception as e:
            raise SandboxTaskFailureError('Unable to fetch Sirena files. \n {} \n {}'.format(traceback.format_exc(), e))
