# coding=utf-8
import logging
import traceback
from datetime import datetime

import sandbox.common.types.misc as ctm
import sandbox.common.types.notification as ctn
from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.shared_flights.ssim_parser_resources import resources
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

PRODUCTION_ENVIRONMENT = 'production'


class AviaSharedFlightsSsimParserTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    """ Parses SSIM files from Innovata """

    class Requirements(sdk2.Task.Requirements):
        # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Import parameters') as import_block:
            ftp_host = sdk2.parameters.String(
                'SSIM files ftp host',
                required=False,
                default_value='ftp.us.cirium.com',
            )
            login = sdk2.parameters.String(
                'SSIM files ftp login',
                required=False,
                default_value='YANDEX',
            )
            password = sdk2.parameters.String(
                'SSIM files ftp password',
                required=False,
                default_value='',
            )
            vaults_owner = sdk2.parameters.String(
                'SSIM files ftp password vault entry owner',
                required=True,
                default_value='AVIA',
            )
            environment = sdk2.parameters.String(
                'Environment (testing or production)',
                required=True,
                default_value='testing',
            )
            new_transport_models_notification_recipients = sdk2.parameters.List(
                'New transport models notification recipients',
                required=False,
                default=[],
                value_type=sdk2.parameters.String,
            )

    def on_prepare(self):
        configure_logging(
            sentry_dsn=get_sentry_dsn(self)
        )
        self._logger = logging.getLogger(__name__)

        super(AviaSharedFlightsSsimParserTask, self).on_prepare()

    def on_execute(self):
        import shutil
        from travel.avia.shared_flights.tasks.ssim_parser.ssim_fetcher import SsimFetcher

        ftp_password = self.Parameters.password if self.Parameters.password \
            else sdk2.Vault.data(self.Parameters.vaults_owner, 'AVIA_FTP_INNOVATA_PASSWORD')

        parsed_at = datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S')

        existing_resources = []
        if self.Parameters.environment == PRODUCTION_ENVIRONMENT:
            existing_resources = list(
                resources.AviaSharedFlightsLastUpdatedProductionResource.find(state=ctr.State.READY).limit(2))
        else:
            existing_resources = list(
                resources.AviaSharedFlightsLastUpdatedTestingResource.find(state=ctr.State.READY).limit(2))

        if len(existing_resources) > 1:
            raise SandboxTaskFailureError(
                'Too many AviaSharedFlightsLastUpdatedResource instances found. '
                'Please manually delete all except one.'
            )
        elif len(existing_resources) == 1:
            resource = existing_resources[0]
        else:
            resource = None
            if self.Parameters.environment == PRODUCTION_ENVIRONMENT:
                resource = resources.AviaSharedFlightsLastUpdatedProductionResource(self, 'Latest processed date', '')
            else:
                resource = resources.AviaSharedFlightsLastUpdatedTestingResource(self, 'Latest processed date', '')
            # Create dummy file, as sandbox doesn't like it when the newly created resource has no files
            resource_data = sdk2.ResourceData(resource)
            resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
            last_parsed_path = resource_data.path.joinpath('last-parsed')
            last_parsed_path.write_bytes('Tracking resource, created at {}'.format(parsed_at))
            resource_data.ready()

        ssim_fetcher = SsimFetcher(
            hostname=self.Parameters.ftp_host,
            username=self.Parameters.login,
            password=ftp_password,
            logger=self._logger,
        )
        try:
            result = ssim_fetcher.fetch(resource.last_parsed_date, None)

            if result is not None:
                # create output resource
                if self.Parameters.environment == PRODUCTION_ENVIRONMENT:
                    output_resource = resources.AviaSharedFlightsInnovataProductionResource(self,
                                                                                            "Parsed innovata files",
                                                                                            "references")
                else:
                    output_resource = resources.AviaSharedFlightsInnovataTestingResource(self, "Parsed innovata files",
                                                                                         "references")
                output_resource.created_at = parsed_at
                output_resource.source_data_date = result.current_ssim_date
                output_resource_data = sdk2.ResourceData(output_resource)
                output_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)

                # copy temp files into the resource files
                aircraft_path = output_resource_data.path.joinpath("aircraft.bin")
                shutil.copyfile(result.aircraft_output_file, aircraft_path.absolute().as_posix())

                flight_bases_path = output_resource_data.path.joinpath("flight_bases.bin")
                shutil.copyfile(result.flight_bases_file, flight_bases_path.absolute().as_posix())

                flight_patterns_path = output_resource_data.path.joinpath("flight_patterns.bin")
                shutil.copyfile(result.flight_patterns_file, flight_patterns_path.absolute().as_posix())

                designated_carriers_path = output_resource_data.path.joinpath("designated_carriers.bin")
                shutil.copyfile(result.designated_carriers_file, designated_carriers_path.absolute().as_posix())

                flying_carriers_path = output_resource_data.path.joinpath("flying_carriers.bin")
                shutil.copyfile(result.flying_carriers_file, flying_carriers_path.absolute().as_posix())

                ssim_codeshares_path = output_resource_data.path.joinpath("codeshares.bin")
                shutil.copyfile(result.codeshares_file, ssim_codeshares_path.absolute().as_posix())

                self._logger.info('Data files are processed, updating the tracking record resource.')

                output_resource_data.ready()
                resource.last_parsed_date = result.current_ssim_date
                resource.parsed_at = parsed_at
                resource.last_parsed_resource_id = output_resource.id

                recipients = self.Parameters.new_transport_models_notification_recipients
                if recipients and result.new_transport_models:
                    self._logger.info(
                        'Reporting new transport models to recipients: %s',
                        recipients,
                    )
                    message_body = '\n'.join(result.new_transport_models)
                    self._logger.info('Message body: %s', message_body)
                    self.server.notification(
                        subject='Новые модели транспорта из Инноваты от {}'.format(datetime.today().strftime('%Y-%m-%d')),
                        body=message_body,
                        recipients=recipients,
                        transport=ctn.Transport.EMAIL,
                        type='plain',
                    )

            self._logger.info('Task has been completed.')
        except Exception as e:
            raise SandboxTaskFailureError('Unable to fetch Innovata files. \n {} \n {}'.format(traceback.format_exc(), e))
