# coding=utf-8
from __future__ import unicode_literals

import copy
import logging
import yaml

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.avia.base import AviaBaseTask

MAX_DEPTH = 10


class AviaSolomonGenerateAlerts(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Task.Requirements):
        # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        project = sdk2.parameters.String('Solomon project name', required=True, default_value='avia')
        token_owner = sdk2.parameters.String(
            'Vault token owner name',
            required=True,
            default_value='AVIA',
        )
        solomon_token_name = sdk2.parameters.String(
            'Solomon vault token name',
            required=True,
            default_value='SOLOMON_TOKEN',
        )
        juggler_token_name = sdk2.parameters.String(
            'Juggler vault token name',
            required=True,
            default_value='JUGGLER_TOKEN',
        )
        config_path = sdk2.parameters.SvnUrl(
            'Path to config in arcadia',
            required=True,
            default_value=Arcadia.trunk_url('travel/avia/configs/solomon/alerts.yaml')
        )
        solomon_api_uri = sdk2.parameters.String(
            'Solomon api URI',
            required=True,
            default_value='http://solomon.yandex.net/api/v2',
        )
        juggler_api_uri = sdk2.parameters.String(
            'Juggler api URI',
            required=True,
            default_value='http://juggler-api.search.yandex.net',
        )

    @staticmethod
    def _get_import_files(import_config):
        if not import_config:
            return
        prefix = import_config.get('prefix', 'svn+ssh://zomb-sandbox-rw@arcadia.yandex.ru')
        directory = import_config.get('directory', '/arc/trunk/arcadia/')
        for filename in import_config.get('files', []):
            yield prefix + directory + filename

    def _build_config(self, config_path, current_depth=0):
        if current_depth > MAX_DEPTH:
            raise ValueError('Too deep config')
        config = yaml.load(Arcadia.cat(config_path))
        for import_config in config.pop('import', []):
            for filename in self._get_import_files(import_config):
                config.update(self._build_config(filename, current_depth + 1))
        return config

    def on_execute(self):
        from juggler_sdk import Check, Child, NotificationOptions, FlapOptions

        from .juggler import AviaJugglerApi
        from .solomon import (SolomonApi, SolomonNotificationChannel, SolomonAlert, SysFsSolomonAlert,
                              PushClientSolomonAlert, NginxSolomonAlert)

        solomon_api = SolomonApi(
            '{}/projects/{}'.format(self.Parameters.solomon_api_uri, self.Parameters.project),
            sdk2.Vault.data(self.Parameters.token_owner, self.Parameters.solomon_token_name),
        )

        juggler_api = AviaJugglerApi(
            self.Parameters.juggler_api_uri,
            sdk2.Vault.data(self.Parameters.token_owner, self.Parameters.juggler_token_name),
            '{}_auto_solomon'.format(self.Parameters.project),
            'travel.avia',
        )

        logging.debug('Config path: %s', self.Parameters.config_path)
        config = self._build_config(self.Parameters.config_path)
        logging.debug('Config: %s', config)

        alert_ids = solomon_api.get_alert_ids()
        notification_channel_ids = solomon_api.get_notification_channel_ids()

        defaults = config.pop('defaults')
        notification_channels = config.pop('notification_channels', {})
        default_juggler = config.pop('juggler', {})

        logging.info('Create solomon notification channels ...')
        for channel_id, channel_config in notification_channels.items():
            channel = SolomonNotificationChannel(
                self.Parameters.project,
                channel_id,
                channel_config['statuses'],
                channel_config['method'],
            )
            if channel_id in notification_channel_ids:
                solomon_api.update_notification_channel(channel_id, channel.prepare())
                logging.info('Channel %s updated', channel_id)
            else:
                solomon_api.create_notification_channel(channel.prepare())
                logging.info('Channel %s created', channel_id)

        for cluster_check_id, cluster_config in config.items():
            logging.info('Processing %s ...', cluster_check_id)
            for service in (SysFsSolomonAlert.SERVICE, PushClientSolomonAlert.SERVICE, NginxSolomonAlert.SERVICE):
                for check in cluster_config.get(service, {}).get('checks', []):
                    logging.info('Create solomon alerts for %s ...', check)
                    check_config = defaults.get(check, {})
                    check_config.update(cluster_config[service].get(check, {}))
                    alert = SolomonAlert.create(
                        service,
                        self.Parameters.project,
                        cluster_check_id,
                        check,
                        cluster_config,
                        check_config,
                    )
                    alert_dict = alert.prepare(check_config)

                    juggler_config = copy.deepcopy(default_juggler)
                    juggler_config.update(check_config.get('juggler', {}))

                    juggler_check = Check(
                        host='{}_{}'.format(self.Parameters.project, cluster_check_id),
                        service=alert.get_juggler_service(),
                        refresh_time=juggler_config.get('refresh_time', 90),
                        ttl=juggler_config.get('ttl', 900),
                        aggregator='logic_or',
                        aggregator_kwargs={
                            'ignore_nodata': True,
                        },
                        children=[
                            Child(
                                group_type='EVENTS',
                                host=alert.get_juggler_events_filter(),
                                service='all',
                                instance='all',
                            )
                        ],
                        notifications=[],
                        tags=[self.Parameters.project, alert.SERVICE, 'solomon'],
                        description=alert_dict['name'],
                    )

                    for notification_config in juggler_config.get('notifications', []):
                        n = NotificationOptions(
                            template_name=notification_config['template_name'],
                            template_kwargs=notification_config['template_kwargs'],
                        )
                        juggler_check.notifications.append(n)

                    flaps_config = juggler_config.get('flaps', [])
                    if flaps_config:
                        juggler_check.flaps_config = FlapOptions(
                            stable=flaps_config['stable_time'],
                            critical=flaps_config['critical_time'],
                        )

                    juggler_api.create_check(juggler_check)
                    logging.info('Juggler check created: %s', juggler_check.service)

                    if alert.id in alert_ids:
                        solomon_api.update_alert(alert.id, alert_dict)
                        alert_ids.remove(alert.id)
                        logging.info('Solomon alert %s updated', alert.id)
                    else:
                        solomon_api.create_alert(alert_dict)
                        logging.info('Solomon alert %s created', alert.id)

        for alert_id in alert_ids:
            solomon_api.remove_alert(alert_id)
            logging.info('Solomon alert %s removed', alert_id)

        juggler_api.cleanup()
        logging.info('Juggler cleaned')

        logging.info('Done')
