# coding=utf-8
from __future__ import unicode_literals

import copy
import re

from requests import Session
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry


class SolomonApi(object):
    # https://solomon.yandex-team.ru/swagger-ui.html

    def __init__(self, api_uri, token):
        self._api_uri = api_uri
        self._token = token
        self._session = Session()
        adapter = HTTPAdapter(
            max_retries=Retry(
                total=3,
                read=3,
                connect=3,
                backoff_factor=0.1,
                status_forcelist=(500, 502, 503, 504),
                method_whitelist=('GET', 'PUT', 'POST'),
            ),
        )
        self._session.mount('http://', adapter)
        self._session.mount('https://', adapter)

    def _request(self, method, path, params=None, json_data=None):
        # type: (str, str, dict, dict) -> dict
        response = self._session.request(
            method,
            '{}{}'.format(self._api_uri, path),
            params=params,
            headers={
                'Authorization': 'OAuth {}'.format(self._token),
            },
            json=json_data,
        )
        response.raise_for_status()

        return response.json() if response.content else None

    def _get_item_ids(self, path, name_prefix=None):
        # type: (str, str) -> set
        ids = set()

        page_token = 0
        while page_token is not None:
            response = self._request('GET', path, {
                'pageSize': 1000,
                'pageToken': page_token,
            })
            page_token = response.get('nextPageToken', None)
            for item in response['items']:
                if not name_prefix or item['name'].startswith(name_prefix):
                    ids.add(item['id'])

        return ids

    def _get_item(self, path, item_id):
        # type: (str, str) -> dict
        return self._request('GET', path + '/{}'.format(item_id))

    def _create_item(self, path, item):
        # type: (str, dict) -> None
        self._request('POST', path, {}, item)

    def _update_item(self, path, item_id, item):
        # type: (str, str, dict) -> None
        item['version'] = self._get_item(path, item_id)['version']
        self._request('PUT', path + '/{}'.format(item_id), {}, item)

    def _remove_item(self, path, item_id):
        self._request('DELETE', path + '/{}'.format(item_id))

    def get_alert_ids(self):
        # type: () -> set
        return self._get_item_ids('/alerts', name_prefix='Avia / Auto /')

    def create_alert(self, alert):
        # type: (dict) -> None
        self._create_item('/alerts', alert)

    def get_alert(self, alert_id):
        # type: (str) -> dict
        return self._get_item('/alerts', alert_id)

    def update_alert(self, alert_id, alert):
        # type: (str, dict) -> None
        self._update_item('/alerts', alert_id, alert)

    def remove_alert(self, alert_id):
        self._remove_item('/alerts', alert_id)

    def get_notification_channel_ids(self):
        # type: () -> set
        return self._get_item_ids('/notificationChannels', name_prefix='Avia / Auto /')

    def create_notification_channel(self, channel):
        # type: (dict) -> None
        self._create_item('/notificationChannels', channel)

    def get_notification_channel(self, channel_id):
        # type: (str) -> dict
        return self._get_item('/notificationChannels', channel_id)

    def update_notification_channel(self, channel_id, channel):
        # type: (str, dict) -> None
        self._update_item('/notificationChannels', channel_id, channel)


class SolomonNotificationChannel(object):
    TEMPLATE = {
        'id': '',
        'projectId': '',
        'name': '',
        'notifyAboutStatuses': [],
        'repeatNotifyDelayMillis': 0,
        'method': {},
    }

    def __init__(self, project, channel_id, statuses, method):
        self._project = project
        self._channel_id = channel_id
        self._statuses = statuses
        self._method = method

    def _generate_name(self):
        # type: () -> str
        return 'Avia / Auto / {}{}'.format(
            self._channel_id, ' / {}'.format(self._method.keys()[0]) if self._method else ''
        )

    def prepare(self):
        # type: () -> dict
        channel = copy.deepcopy(self.TEMPLATE)
        channel['id'] = self._channel_id
        channel['projectId'] = self._project
        channel['name'] = self._generate_name()
        channel['notifyAboutStatuses'] = self._statuses
        channel['method'] = self._method

        return channel


class SolomonAlert(object):
    SERVICE = None
    GROUP_BY_LABELS = ['host']
    ANNOTATIONS = {
        'trafficLight.color': '{{expression.trafficColor}}',
    }
    TEMPLATE = {
        'id': None,
        'projectId': '',
        'name': '',
        'groupByLabels': [],
        'notificationChannels': [],
        'type': {
            'expression': {
                'program': '',
                'checkExpression': 'is_yellow',
            }
        },
        'annotations': {
            # https://st.yandex-team.ru/SOLOMON-3167
        },
        'periodMillis': 60000,
        'delaySeconds': 10,
    }

    def __init__(self, alert_id, check, project, cluster, http_host='*'):
        self._id = alert_id
        self._check = check
        self._project = project
        self._cluster = cluster
        self._http_host = http_host

    @staticmethod
    def create(service, project, check_id, check, cluster_config, check_config):
        if service == SysFsSolomonAlert.SERVICE:
            return SysFsSolomonAlert(
                '{}_{}_{}'.format(project, check_id, check),
                check,
                project,
                cluster_config['cluster'],
            )
        elif service == PushClientSolomonAlert.SERVICE:
            return PushClientSolomonAlert(
                '{}_{}_{}'.format(project, check_id, check),
                check,
                project,
                cluster_config['cluster'],
            )
        elif service == NginxSolomonAlert.SERVICE:
            return NginxSolomonAlert(
                '{}_{}_{}'.format(project, check_id, check),
                check,
                project,
                cluster_config['cluster'],
                check_config.get('http_host', cluster_config[service].get('http_host', '*')),
                cluster_config[service].get('exclude_uri', []),
            )
        else:
            raise RuntimeError('Unknown service: {}'.format(service))

    @property
    def id(self):
        return self._id

    def _generate_name(self, config):
        return '{project} / Auto / {cluster} / {service} / {check}'.format(
            project=self._project.capitalize(),
            cluster=self._cluster,
            service=self.SERVICE,
            check=re.sub(' +', ' ', '{application} {environment} {component} {http_host} {uri} {check}'.format(
                application=config.get('application', '') if self._http_host == '*' else '',
                component=config.get('component', '') if self._http_host == '*' else '',
                environment=config.get('environment', '') if self._http_host == '*' else '',
                http_host=self._http_host if self._http_host != '*' else '',
                uri=config.get('uri', '').replace('/', '_').replace('.', '_'),
                check=self._check,
            )).strip(),
        )

    def _generate_program(self, config):
        return ''

    def prepare(self, config):
        alert = copy.deepcopy(self.TEMPLATE)
        alert['id'] = self._id
        alert['projectId'] = self._project
        alert['name'] = self._generate_name(config)
        alert['groupByLabels'] = self.GROUP_BY_LABELS

        alert['type']['expression']['program'] = self._generate_program(config)

        alert['annotations'].update(self.ANNOTATIONS)
        alert['annotations'].update(config.get('annotations', {}))

        alert['notificationChannels'] = config.get('notification_channels', [])

        return alert

    def get_juggler_service(self):
        return '{}_{}'.format(self.SERVICE, self._check)

    def get_juggler_events_filter(self):
        return '(tag={} & tag={} & tag={} & service={}*)'.format(
            self._project, self.SERVICE, 'solomon', self._id
        )


class NginxSolomonAlert(SolomonAlert):
    SERVICE = 'nginx'
    GROUP_BY_LABELS = ['host', 'method', 'uri', 'application', 'environment', 'component', 'http_host']
    STATUS_CHECK_PROGRAM = (
        "let is_red = false;\n"
        "let is_yellow = false;\n\n"
        "let countFiltered = group_lines('sum', {{project='{project}', cluster='{cluster}', service='{service}', "
        "application='{application}', environment='{environment}', component='{component}', "
        "sensor='count', status='{status}', {uri}, method='{method}', host!='_all', http_host='{http_host}'}});\n"
        "let countAll = group_lines('sum', {{project='{project}', cluster='{cluster}', service='{service}', "
        "application='{application}', environment='{environment}', component='{component}', "
        "sensor='count', status='2xx|3xx|4xx|5xx', {uri}, method='{method}', host!='_all', http_host='{http_host}'}});\n"
        "let percent = min(countFiltered / countAll) * 100;\n"
        "let is_red = is_red || percent > {crit_threshold};\n"
        "let is_yellow = is_yellow || percent > {warn_threshold};\n\n"
        "let trafficColor = is_red ? 'red' : (is_yellow ? 'yellow' : 'green');\n"
        "let description = is_red ? 'more than {crit_threshold}%' : (is_yellow ? 'more than {warn_threshold}%' : 'OK');\n"
    )

    def __init__(self, alert_id, check, project, cluster, http_host='*', exclude_uri=None):
        super(NginxSolomonAlert, self).__init__(alert_id, check, project, cluster, http_host)
        if exclude_uri:
            self._exclude_uri = exclude_uri if isinstance(exclude_uri, list) else list(exclude_uri)
        else:
            self._exclude_uri = []

    def _generate_uri(self, uri, exclude_uri):
        if self._exclude_uri or exclude_uri:
            if not isinstance(exclude_uri, list):
                exclude_uri = list(exclude_uri)
            return 'uri!=\'{}\''.format('|'.join(self._exclude_uri + exclude_uri))

        return 'uri=\'{}\''.format(uri)

    def _generate_program(self, config):
        if config['type'] == 'status_check':
            program = self.STATUS_CHECK_PROGRAM.format(
                service=self.SERVICE,
                cluster=self._cluster,
                project=self._project,
                application=config.get('application', '*'),
                status=config['status'],
                environment=config.get('environment', '*'),
                component=config.get('component', '*'),
                uri=self._generate_uri(config.get('uri', '*'), config.get('exclude_uri', [])),
                http_host=self._http_host,
                method=config.get('method', '*'),
                crit_threshold=config['checks']['crit'],
                warn_threshold=config['checks']['warn'],
            )

            return program

        raise RuntimeError('Unknown type: {}'.format(config['type']))


class SysFsSolomonAlert(SolomonAlert):
    SERVICE = 'sys-fs'
    GROUP_BY_LABELS = ['host', 'instance', 'mount_point']
    ANNOTATIONS = {
        'trafficLight.color': '{{expression.trafficColor}}',
        'percentFree': '{{expression.percent_free_str}}',
        'percentUsed': '{{expression.percent_used_str}}',
    }
    STATUS_CHECK_PROGRAM = (
        "let is_red = false;\n"
        "let is_yellow = false;\n\n"
        "let diskSize = avg({{project='{project}', cluster='{cluster}', service='{service}', "
        "sensor='disk-size', host!='_all', mount_point='*'}});\n"
        "let diskFree = avg({{project='{project}', cluster='{cluster}', service='{service}', "
        "sensor='disk-free', host!='_all', mount_point='*'}});\n"
        "let percent_free = diskFree / diskSize * 100;\n"
        "let percent_used = (diskSize - diskFree) / diskSize * 100;\n"
        "let is_red = is_red || percent_free < {crit_threshold};\n"
        "let is_yellow = is_yellow || percent_free < {warn_threshold};\n\n"
        "let trafficColor = is_red ? 'red' : (is_yellow ? 'yellow' : 'green');\n"
        "let description = is_red ? 'disk free less then {crit_threshold}%' : (is_yellow ? 'disk free less then {warn_threshold}%' : 'OK');\n"
        "let percent_free_str = to_fixed(percent_free, 0);\n"
        "let percent_used_str = to_fixed(percent_used, 0);\n"
    )

    def __init__(self, alert_id, check, project, cluster):
        super(SysFsSolomonAlert, self).__init__(alert_id, check, project, cluster)

    def _generate_program(self, config):
        if config['type'] == 'disk_free':
            program = self.STATUS_CHECK_PROGRAM.format(
                service=self.SERVICE,
                cluster=self._cluster,
                project=self._project,
                crit_threshold=config['checks']['crit'],
                warn_threshold=config['checks']['warn'],
            )

            return program

        raise RuntimeError('Unknown type: {}'.format(config['type']))


class PushClientSolomonAlert(SolomonAlert):
    SERVICE = 'push-client'
    GROUP_BY_LABELS = ['host', 'instance', 'log_name']
    ANNOTATIONS = {
        'trafficLight.color': '{{expression.trafficColor}}',
        'sensor_value': '{{expression.sensor_value}}',
    }
    STATUS_CHECK_PROGRAM = (
        "let is_red = false;\n"
        "let is_yellow = false;\n\n"
        "let sensor_value = last({{project='{project}', cluster='{cluster}', service='{service}', "
        "sensor='{sensor}', host!='_all', log_name='*'}});\n"
        "let lag_value = last({{project='{project}', cluster='{cluster}', service='{service}', sensor='lag', "
        "host!='_all', log_name='*'}});\n"
        "let is_red = is_red || (sensor_value >= {crit_threshold} && lag_value > 0);\n"
        "let is_yellow = is_yellow || (sensor_value >= {warn_threshold} && lag_value > 0);\n\n"
        "let trafficColor = is_red ? 'red' : (is_yellow ? 'yellow' : 'green');\n"
        "let description = is_red ? '{sensor} more than {crit_threshold}' : (is_yellow ? '{sensor} more than {warn_threshold}' : 'OK');\n"
    )
    CHECKS = {
        'push_client_delay': 'commit-delay',
        'push_client_lag': 'lag',
    }

    def __init__(self, alert_id, check, project, cluster):
        super(PushClientSolomonAlert, self).__init__(alert_id, check, project, cluster)

    def _generate_program(self, config):
        if config['type'] in self.CHECKS:
            program = self.STATUS_CHECK_PROGRAM.format(
                service=self.SERVICE,
                cluster=self._cluster,
                project=self._project,
                crit_threshold=config['checks']['crit'],
                warn_threshold=config['checks']['warn'],
                sensor=self.CHECKS[config['type']],
            )

            return program

        raise RuntimeError('Unknown type: {}'.format(config['type']))
