import logging
import os
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib import statface_helpers, yql_helpers


YESTERDAY = (datetime.now() - timedelta(days=1)).strftime('%Y-%m-%d')


class AviaSovetnikPriceComparison(AviaBaseTask):
    """
    Compare our and competitors prices via Sovetnik
    See https://st.yandex-team.ru/RASPTICKETS-13617
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('requests'),
            PipEnvironment('yql'),
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Date settings') as date_block:
            left_date = sdk2.parameters.String('Start date (deafult yesterday)', required=False, default='')
            right_date = sdk2.parameters.String('End date (deafult yesterday)', required=False, default='')

        with sdk2.parameters.Group('Statface settings') as statface_block:
            stat_path = sdk2.parameters.String('Statface path', required=True)
            stat_beta = sdk2.parameters.Bool('Use beta version of Statface', default=False, required=True)

        with sdk2.parameters.Group('Vault settings') as vault_block:
            user = sdk2.parameters.String('User', required=True)
            statface_token_vault = sdk2.parameters.String('Name of the vault with Statface token', required=True)
            yt_token_vault = sdk2.parameters.String('Name of the vault with YT token', required=True)
            yql_token_vault = sdk2.parameters.String('Name of the vault with YQL token', required=True)

    def _read_query(self):
        with open(os.path.join(os.path.dirname(__file__), 'query.sql')) as inp:
            return inp.read()

    def _get_token(self, system):
        if system == 'yt':
            vault_name = self.Parameters.yt_token_vault

        elif system == 'yql':
            vault_name = self.Parameters.yql_token_vault

        elif system == 'statface':
            vault_name = self.Parameters.statface_token_vault

        else:
            raise ValueError('Unknown system: {}'.format(system))

        return sdk2.Vault.data(self.Parameters.user, vault_name)

    def on_execute(self):
        import yt.wrapper as yt
        from yql.api.v1.client import YqlClient

        logging.info('Start')
        yt_client = yt.YtClient(config={
            'token': self._get_token('yt'),
            'proxy': {'url': 'hahn'},
        })

        yql_client = YqlClient(token=self._get_token('yql'))

        temp_table = yt_client.create_temp_table()
        logging.info('Output table: %s', temp_table)
        query = self._read_query().format(
            output_table='`{}`'.format(temp_table),
            left_date=self.Parameters.left_date or YESTERDAY,
            right_date=self.Parameters.right_date or YESTERDAY,
        )
        r = yql_client.query(query, syntax_version=1)
        r.run()
        logging.info('YQL Operation: %s', yql_helpers.get_yql_operation_url(r))
        r.wait_progress()

        if not r.is_success:
            logging.error('Operation failed. Status:', r.status)
            if r.errors:
                for error in r.errors:
                    logging.error(' - %s', str(error))
            raise SandboxTaskFailureError('YQL query failed')

        records = list(yt_client.read_table(temp_table, format=yt.JsonFormat()))
        r = statface_helpers.post_data_to_stat(
            name=self.Parameters.stat_path,
            data=records,
            token=self._get_token('statface'),
            beta=self.Parameters.stat_beta,
        )

        if r.status_code != 200:
            raise SandboxTaskFailureError(
                'Uploading to stat failed. Code: {}, message: {}'.format(
                    r.status_code,
                    r.content,
                )
            )
