# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import shutil
import tarfile
from pathlib2 import Path
from sandbox import sdk2
from sandbox.projects.avia.travel_avia_dump_resource.task.secret_helper import resolve_args_secrets, \
    resolve_environment_secrets, secrets_help
from sandbox.projects.common import binary_task

log = logging.getLogger()


class TravelAviaDumpResource(binary_task.LastRefreshableBinary, sdk2.Task):
    """
    Универсальная задача, которая запускает исполняемый файл,
    а затем собирает артифакты в sandbox ресурсы
    """

    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 1024  # 1024 Megs
        ram = 8 * 1024  # 8 Gb

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(binary_task.LastBinaryReleaseParameters):

        yav_delegate_helper = sdk2.parameters.YavSecret(
            'Yav delegate helper',
            description='This field helps delegate yav secrets to sandbox. '
                        'It\'s not and should not be used in actual task code',
        )

        with sdk2.parameters.Group('Executable') as executable_params:
            executable = sdk2.parameters.Resource('Resource with executable', required=True)
            executable_path = sdk2.parameters.String('Path to executable in resource tar file', required=True)

        with sdk2.parameters.Group('Parameters and arguments') as run_parameters:
            run_environment = sdk2.parameters.Dict('Run with this environments', description=secrets_help)
            run_args = sdk2.parameters.List('Run with this arguments', description=secrets_help)

        output_path = sdk2.parameters.String('Path to generated resources', default='./output')

    def on_execute(self):
        super(TravelAviaDumpResource, self).on_execute()
        self._extract_from_resource(
            resource=self.Parameters.executable,
            path_to_member=self.Parameters.executable_path,
        )

        output_dir = Path(self.Parameters.output_path)
        output_dir.mkdir(parents=True, exist_ok=True)
        log.info('Created directory {}'.format(output_dir.absolute()))

        args = resolve_args_secrets(self.Parameters.run_args)
        environment = resolve_environment_secrets(self.Parameters.run_environment)

        self._run_dumper(args, environment)

        output_files = [p for p in output_dir.iterdir() if p.is_file()]
        log.info('The following output files are detected: {}'.format(output_files))
        for output_file in output_files:
            try:
                self._upload_resource(output_file)
            except Exception:
                log.exception('Error while uploading %s', output_file)

        log.info('Done')

    @classmethod
    def _extract_from_resource(cls, resource, path_to_member):
        data = sdk2.ResourceData(resource)
        data_path = data.path.absolute().as_posix()
        tar_file = tarfile.open(data_path)
        tar_file.extract(member=path_to_member)
        log.info('Extracted {} from {}'.format(path_to_member, data_path))

    def _run_dumper(self, params, environment_vars):
        log.info('Running dumper')
        dumper_path = './{}'.format(self.Parameters.executable_path)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('dumper')) as pl:
            return_code = sdk2.helpers.subprocess.Popen(
                [dumper_path] + params,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=environment_vars,
            ).wait()
            if return_code != 0:
                error_message = 'Child dumper process has returned non-zero exit code: {}'.format(return_code)
                log.error(error_message)
                raise Exception(error_message)

        log.info('Dumper stopped')

    @staticmethod
    def _load_resources():
        from sandbox.projects.Travel.resources import dicts

        result = {}
        for name, entry in dicts.__dict__.items():
            if not hasattr(entry, 'resource_name'):
                continue
            if dicts.is_avia_dict(entry):
                result[entry.resource_name] = entry
        return result

    def _get_resource_by_name(self, base_name):
        name, _ = base_name.split('.')
        if not hasattr(self, '_resources'):
            self._resources = self._load_resources()

        return self._resources.get(name)

    def _upload_resource(self, output_file):
        output_file_absolute_path = output_file.absolute()
        log.info('Trying to upload {}'.format(output_file_absolute_path))

        base_name = output_file.name
        resource_class = self._get_resource_by_name(base_name)
        if resource_class is None:
            log.error('Can\'t find resource_class for resource name %s', base_name)
            return

        resource = resource_class(self, description=base_name, path=base_name.replace('.bin', '.data'), ttl='inf')
        resource_data = sdk2.ResourceData(resource)
        shutil.copy(
            src=output_file_absolute_path.as_posix(),
            dst=resource_data.path.absolute().as_posix(),
        )
        resource_data.ready()
        log.info('File {} uploaded to resource'.format(output_file_absolute_path))
